<?php

/**
 * @file
 * Defines configuration classes for the Autolink module.
 */

require_once(drupal_get_path('module', 'autolink') .'/includes/autolink.info.inc');

/**
 * Defines the configuration class.
 *
 * The configuration class extends AutolinkInfo and thus has access to
 * all types of information. It can be used to retrieve configuration
 * information based on plugin settings forms. Plugin settings forms
 * are defined in children of the AutolinkConfigForm class.
 *
 * @ingroup autolink
 */
class AutolinkConfig extends AutolinkInfo {
  const DISABLED = 0;
  const ENABLED = 1;
  public $_plugin = NULL;
  public $link_types = array();
  public $plugins = array();
  protected $info = array();
  public $settings = array();
  protected $form;

  /**
   * Sets properties by querying the information manager class for data.
   */
  function __construct() {
    $this->load();
    $this->link_types = autolink_get('LinkTypeInfo')->getAll();
    $this->plugins = autolink_get('PluginInfo')->getAll();
    $this->info['plugin'] = autolink_get('PluginInfo')->getCurrent();
    // Populate enabled link types.
    $this->info['link_types'] = array();
    foreach ($this->link_types as $key => $value) {
      if (isset($this->settings['link_types'][$key]) && $this->settings['link_types'][$key] == self::ENABLED && in_array($key, $this->info['plugin']['link types'])) {
        $this->info['link_types'][$key] = $value;
      }
    }
  }

  /**
   * Sets a form to be used in configuration.
   *
   * @param $plugin
   *   An optional string identifying a specific plugin to use. If no
   *   plugin is specified then the currently enabled plugin will be used.
   * @return
   *   An AutolinkConfigForm object based on the class defined by
   *   the currently enabled plugin. The object is first stored in
   *   the $form property and returned here for continued method calls.
   */
  function setForm($plugin = NULL) {
    $info = !empty($plugin) ? autolink_get('PluginInfo')->getPlugin($plugin) : autolink_get('PluginInfo')->getCurrent();
    if (!empty($info['config']['class'])) {
      $class = $info['config']['class'];
      if (!empty($info['config']['file'])) {
        try {
          $module = isset($info['config']['module']) ? $info['config']['module'] : $info['module'];
          Autolink::includeFile($module, $info['config']['file']);
        }
        catch (AutolinkException $e) {
          $e->displayMessage();
        }
      }
    }
    else {
      $class = 'AutolinkConfigForm';
    }
    if (class_exists($class)) {
      $this->form = new $class($this);
      return $this->form;
    }
    return FALSE;
  }

  /**
   * Resets Autolink settings to defaults.
   */
  function reset() {
    $this->settings = array();
  }

  /**
   * Loads the settings array from Drupal variables.
   */
  private function load() {
    $this->_plugin = variable_get('autolink_plugin', 'autolink');
    $this->settings = variable_get('autolink_config_'. $this->_plugin, array());
  }

  /**
   * Returns a single settings object.
   *
   * @param $var
   *   The setting which should be retrieved.
   * @param $default
   *   A default setting which will be returned if no setting is found.
   *   The is exactly the same as using variable_get().
   *
   * @return
   *   An AutolinkSingleSetting object.
   *
   * @see AutolinkSingleSetting
   */
  function fetchSetting($var, $default = self::ENABLED) {
    $data = isset($this->settings[$var]) ? $this->setting[$var] : $default;
    return new AutolinkSingleSetting($var, $data, $this);
  }

  /**
   * Returns an array setting object. This should only be used for settings
   * that create arrays like checkboxes or multiple select lists.
   *
   * @param $var
   *   A string representing the setting to return. For plugin settings,
   *   this string directly correlates to the form item's key in the same
   *   way as is done with system_settings_form().
   *
   * @see AutolinkArraySetting
   */
  function fetchArraySetting($var) {
    $data = isset($this->settings[$var]) ? $this->settings[$var] : array();
    return new AutolinkArraySetting($var, $data, $this);
  }

  /**
   * Returns a single setting value without instantiating an object.
   *
   * @param $var
   *   The setting to return. For plugins, this string directly correlates
   *   to the form item's key in the same way as is done with system_settings_form().
   * @param $default
   *   A default value to return if no setting is found.
   */
  function get($var, $default = 0) {
    return isset($this->settings[$var]) ? $this->settings[$var] : $default;
  }

  /**
   * Sets a Drupal variable from arguments.
   *
   * @param $var
   *   The setting to set.
   * @param $value
   *   The value to assign to the indicated setting.
   */
  function set($var, $value) {
    $this->settings[$var] = $value;
  }

  /**
   * Deletes a Drupal variable.
   */
  function delete($var) {
    unset($this->settings[$var]);
  }

  /**
   * Saves plugins and configuration settings as an array in Drupal variables.
   */
  function save() {
    // Save the configuration.
    variable_set('autolink_config_'. $this->_plugin, $this->settings);
    // Make a note that settings for this plugin have been saved.
    $configs = variable_get('autolink_configurations', array());
    $configs[$this->_plugin] = $this->_plugin;
    variable_set('autolink_configurations', $configs);
    // If the plugin has changed, update the plugin and notify modules.
    if ($this->_plugin != $this->get('plugin', AUTOLINK_DEFAULT_PLUGIN)) {
      module_invoke_all('autolink_plugin_'. $this->_plugin .'_disable');
      $this->_plugin = $this->get('plugin', AUTOLINK_DEFAULT_PLUGIN);
      variable_set('autolink_plugin', $this->_plugin);
      module_invoke_all('autolink_plugin_'. $this->_plugin .'_enable');
      // The menu_rebuild_needed variable is set to ensure that the menu
      // gets rebuilt *after* our variables are set. Otherwise, the menu
      // is rebuilt before Autolink settings are set and thus the Autolink
      // links menu item does not get created in certain situations.
      variable_set('menu_rebuild_needed', TRUE);
      // Ensure that all node types' links are reset.
      variable_set('autolink_plugin_reset', node_get_types('names'));
    }
    parent::resetCache();
  }

  /**
   * Returns a specific array of settings from Autolink settings.
   *
   * This function is built to handle the unique way Autolink stores
   * multiple value variables. It was built as a simpler method to
   * retrieve settings for all of a certain variable type, for instance
   * it retrieves different types of node type settings. The function
   * retrieves a list of settings and populates a return array based on
   * parameters. Warning: the default setting that is used for each
   * value of the array is always 0, as defined in AutolinkArraySetting.
   *
   * @param $var
   *   A string representing a specific setting to return.
   * @param $array
   *   An array with keys to use to get variables.
   *
   * @return
   *   A copy of the $array parameter but with disabled settings removed.
   */
  public function getSettings($var, array $array) {
    $settings = array();
    $setting = $this->fetchArraySetting($var);
    foreach ($array as $key => $value) {
      if ($setting->isEnabled($key)) {
        $settings[$key] = $value;
      }
    }
    return $settings;
  }

  /**
   * Returns all link types that have been defined.
   */
  function getLinkTypes() {
    return $this->link_types;
  }

  /**
   * Returns link types that are enabled and supported by the enabled plugin.
   */
  function getAvailableLinkTypes() {
    return $this->info['link_types'];
  }

  /**
   * Returns available link types in a key => label pair for display in lists.
   */
  static function displayAvailableLinkTypes() {
    $types = array();
    foreach ($this->link_types as $key => $value) {
      if ($this->settings['link_types'][$key] == ENABLED && in_array($key, $this->info['plugin']['link types'])) {
        $this->info['link_types'][$key] = $value;
      }
    }
  }

  /**
   * Returns data about an individual link type.
   */
  function getLinkType($type) {
    return $this->link_types[$type];
  }

  /**
   * Returns information on all defined plugins.
   */
  function getPlugins() {
    return $this->plugins;
  }

  /**
   * Returns the currently enabled plugin information.
   */
  function getPlugin() {
    return $this->setings['plugin'];
  }

}

/**
 * Defines the configuration form interface.
 */
interface AutolinkConfigFormInterface {
  function execute($method, array &$a2, array &$a3);
}

/**
 * Defines the configuration form class.
 *
 * This class uses the magic __call() method to call methods that are
 * within the attached configuration object. Also available are magic
 * __get() and __set() methods, so most configuration object properties
 * and methods are available in public scope.
 *
 * @ingroup forms
 * @ingroup autolink
 */
class AutolinkConfigForm implements AutolinkConfigFormInterface {
  private $config;

  /**
   * Constructor method inserts current configuration object.
   */
  function __construct(AutolinkConfig $config) {
    $this->config = $config;
  }

  /**
   * Call method to call configuration object methods.
   */
  function __call($method, $args) {
    if (method_exists($this->config, $method)) {
      return call_user_func_array(array($this->config, $method), $args);
    }
  }

  /**
   * Magic getter method for retrieving properties of the configuration object.
   */
  function __get($property) {
    return $this->config->$property;
  }

  /**
   * Magic setter method sets properties in configuration object.
   */
  function __set($property, $value) {
    return $this->config->$property = $value;
  }

  /**
   * Executes form callbacks for the settings form.
   */
  function execute($method, array &$a2, array &$a3) {
    $method = 'do'. drupal_ucfirst($method);
    if (method_exists($this, $method)) {
      return $this->$method($a2, $a3);
    }
  }

  /**
   * Returns the settings form along with the plugin settings form.
   */
  private function doForm(&$form_state, $settings) {
    $form = $this->getForm($form_state, $settings);
    if (method_exists($this, 'form')) {
      $form += $this->form($form_state, $settings);
    }
    $form += $this->getFormSubmitElements();
    return $form;
  }

  /**
   * Provides the default Autolink settings form.
   */
  private function getForm(&$form_state, $settings) {
    $form = array();
    $form['general_settings'] = array(
      '#type' => 'fieldset',
      '#title' => t('General settings'),
      '#collapsible' => TRUE,
    );

    // Plugin selector.
    $plugins = array();
    $plugin_info = $this->getPlugins();
    $current_plugin = variable_get('autolink_plugin', 'autolink');
    foreach ($plugin_info as $key => $info) {
      $plugins[$key] = $info['label'];
    }

    // Add a filter help description if the plugin uses the filter method.
    if ($plugin_info[$current_plugin]['method'] == 'filter') {
      $form['general_settings']['#description'] = t('%plugin uses a filter to generate content. To set up the Autolink plugin filter please visit the <a href="!filters">filter administration page</a>.', array('%plugin' => $plugin_info[$current_plugin]['label'], '!filters' => url('admin/settings/filters')));
    }

    $form['general_settings']['plugin'] = array(
      '#type' => 'select',
      '#title' => t('Enabled plugin'),
      '#options' => $plugins,
      '#default_value' => $current_plugin,
    );

    // Link types are specific to each plugin.
    $available_types = array();
    if (is_array($this->plugins[$current_plugin])) {
      foreach ($this->plugins[$current_plugin]['link types'] as $type) {
        if ($this->link_types[$type]['identifiable'] == AUTOLINK_ENABLED) {
          $available_types[$type] = $this->link_types[$type]['label'];
        }
      }
    }

    if (!empty($available_types)) {
      $form['general_settings']['link_types'] = array(
        '#type' => 'checkboxes',
        '#title' => t('Link types'),
        '#options' => $available_types,
        '#default_value' => array_keys($this->getSettings('link_types', $available_types)),
        '#description' => t('Note: disabling a link type will remove all defined links from that link type.'),
      );
    }
    return $form;
  }

  /**
   * Returns form submit elements for placement after plugin forms are added.
   */
  private function getFormSubmitElements() {
    $form = array();
    $form['advanced'] = array(
      '#type' => 'fieldset',
      '#title' => t('Advanced settings'),
      '#collapsible' => TRUE,
      '#collapsed' => TRUE,
      '#description' => t('<i>Warning: Advanced settings should only be manipulated by experienced users.</i>'),
    );
    $form['advanced']['development_mode'] = array(
      '#type' => 'checkbox',
      '#title' => t('Enable development mode'),
      '#default_value' => variable_get('autolink_development_mode', 0),
      '#description' => t('Enable development mode to display warning about misconfigured processor, plugin, or link type information or to identify inconsistencies.'),
    );
    $form['advanced']['regex'] = array(
      '#type' => 'checkbox',
      '#title' => t('Use a custom regular expression'),
      '#default_value' => $this->get('regex', 0),
      '#description' => t('Note that this will override any regular expression settings provided by the enabled plugin.'),
    );
    $form['advanced']['regular_expression'] = array(
      '#type' => 'textfield',
      '#title' => t('Custom regular expression'),
      '#default_value' => $this->get('regular_expression', ''),
      '#description' => t('Customize the regular expression used to search for links. Use <strong>[keyword]</strong> to indicate the location of the keyword. Regular expressions should end with a slash <i>/</i>. Autolink will add case-insensitivity <strong>i</strong> and <strong>e</strong> modifiers. Note that Autolink uses the <strong>$1</strong> variable in the replacement argument. $this->buildReplacement($link, "$1").'),
    );
    $form['submit'] = array(
      '#type' => 'submit',
      '#value' => t('Save configuration'),
    );
    $form['buttons']['reset'] = array(
      '#type' => 'submit',
      '#value' => t('Reset to defaults'),
      '#weight' => 10,
    );
    return $form;
  }

  /**
   * Executes form validation callbacks through client code.
   */
  private function doValidate($form, &$form_state) {
    if (method_exists($this, 'validate')) {
      $this->validate($form, &$form_state);
    }
    $this->validateForm($form, $form_state);
  }

  /**
   * Validates the Autolink configuration form.
   */
  private function validateForm($form, &$form_state) {
    if (!empty($form_state['values']['regular_expression'])) {
      $expression = $form_state['values']['regular_expression'];
      // Check that the expression begins and ends with slashes.
      if (drupal_substr($expression, 0, 1) !== '/') {
        form_set_error('regular_expression', t('Regular expressions must begin with a \'/\' character.'));
      }
      if (drupal_substr($expression, -1) !== '/') {
        form_set_error('regular_expression', t('Regular expressions must end with a \'/\' character.'));
      }
      if (strpos($expression, '[keyword]') == FALSE) {
        form_set_error('regular_expression', t('Regular expressions must contain the <i>[keyword]</i> placeholder.'));
      }
    }
  }

  /**
   * Executes form submit callbacks.
   */
  private function doSubmit($form, &$form_state) {
    if (method_exists($this, 'submit')) {
      $this->submit($form, &$form_state);
    }
    $this->submitForm($form, $form_state);
    $this->save();
    AutolinkInfo::resetCache();
  }

  /**
   * Submit handler for the settings form.
   *
   * All settings form elements are stored in a single array as a
   * Drupal variable.
   */
  private function submitForm($form, &$form_state) {
    Autolink::loadFiles(array('links'));
    $op = isset($form_state['values']['op']) ? $form_state['values']['op'] : '';

    // Exclude unnecessary elements.
    unset($form_state['values']['submit'], $form_state['values']['reset'], $form_state['values']['form_id'], $form_state['values']['op'], $form_state['values']['form_token'], $form_state['values']['form_build_id']);

    if ($op == t('Reset to defaults')) {
      $this->reset();
    }
    else {
      variable_set('autolink_development_mode', $form_state['values']['development_mode']);
      unset($form_state['values']['development_mode']);
      foreach ($form_state['values'] as $key => $value) {
        // Set an array for checkbox values. Enabled = 1. Disabled = 0.
        if (is_array($form_state['values'][$key])) {
          $values = array();
          foreach ($form_state['values'][$key] as $type => $state) {
            if ($state) {
              $values[$type] = 1;
            }
            else {
              // Delete links of a certain type when that type becomes disabled.
              if ($key == 'link_types' && $this->fetchArraySetting('link_types')->isEnabled($type)) {
                $result = db_query("SELECT lid FROM {autolink_link} WHERE type = '%s'", $type);
                while ($lid = db_result($result)) {
                  AutolinkLinkMapper::delete($lid);
                }
              }
              $values[$type] = 0;
            }
          }
          $this->set($key, $values);
        }
        else {
          $this->set($key, $value);
        }
      }
    }

    if ($op == t('Reset to defaults')) {
      drupal_set_message(t('The configuration options have been reset to their default values.'));
    }
    else {
      drupal_set_message(t('The configuration options have been saved.'));
    }
  }

  /**
   * The following functions are used in child classes for form handling.
   * This was chosen over an abstract class because they are not required
   * to implement for the enabled plugin.
   */

  /**
   * Defines the additional form elements to build the settings form.
   *
   * The $settings array is usually populated with nothing, as all of the
   * default settings on the form are saved as variables.
   */
  protected function form(&$form_state, $settings) { return array(); }

  /**
   * Validation handler for the settings form. Works as any other validator.
   */
  protected function validate($form, &$form_state) { }

  /**
   * Submit handler for the settings form.
   *
   * If the plugin does not require any additional handling of form data,
   * Autolink will automatically store the data in variables. For checkboxes,
   * Autolink will store each value in its own variable. This submit handler
   * is executed before Autolink's submit handler, so plugins should unset
   * any elements that should not be saved as variables.
   */
  protected function submit($form, &$form_state) { }
}

/**
 * Defines the setting object interface.
 */
interface AutolinkSettingInterface {
  function setDefault($value);
}

/**
 * Defines the Autolink single value setting class.
 */
abstract class AutolinkSetting implements AutolinkSettingInterface {
  protected $config;
  public $name = NULL;
  protected $_data;
  private $default = 0;

  function __construct($name, $data, AutolinkConfig $config) {
    $this->name = $name;
    $this->_data = $data;
    $this->config = $config;
  }

  function __tostring() {
    return is_array($this->_data) ? implode(' ', $this->_data) : $this->_data;
  }

  /**
   * Sets default values for settings that have not yet been set.
   */
  function setDefault($value) {
    $this->default = $value;
  }

  /**
   * prints settings data as is.
   */
  function printData() {
    return $this->_data;
  }
}

/**
 * Defines the scalar setting interface.
 */
interface AutolinkSingleSettingInterface {
  function set($value);
  function delete();
  function save();
  function isEnabled();
}

/**
 * Defines the Autolink single setting class.
 */
class AutolinkSingleSetting extends AutolinkSetting implements AutolinkSingleSettingInterface {

  /**
   * Sets the value of the setting.
   */
  function set($value) {
    $this->_data = $value;
    return $this;
  }

  /**
   * Deletes the setting.
   */
  function delete() {
    $this->config->delete($this->name);
    $this->config->save();
  }

  /**
   * Saves the settings object.
   */
  function save() {
    $this->config->set($this->name, $this->_data);
    $this->config->save();
  }

  /**
   * Returns a boolean value indicating whether the setting is enabled.
   */
  function isEnabled() {
    return $this->_data == 1;
  }
}

/**
 * Defines the array setting interface.
 */
interface AutolinkArraySettingInterface {
  function set($field, $value);
  function delete($field);
  function save();
  function get($field);
  function isEnabled($field);
  function getEnabled();
}

/**
 * Defines the Autolink settings object class.
 *
 * This class supports the use and display of a specific setting or settings
 * array that is retrieved via the AutolinkConfig::fetch() methods.
 */
class AutolinkArraySetting extends AutolinkSetting implements AutolinkArraySettingInterface {

  /**
   * Sets a value of an array.
   */
  function set($field, $value) {
    $this->_data[$field] = $value;
    return $this;
  }

  /**
   * Delets an array value.
   */
  function delete($field) {
    unset($this->_data[$field]);
    return $this;
  }

  /**
   * Saves new settings by updating the configuration object.
   */
  function save() {
    $this->config->set($this->name, $this->_data);
    $this->config->save();
  }

  /**
   * Returns a specific field of array type settings.
   */
  function get($field) {
    return isset($this->_data[$field]) ? $this->_data[$field] : $this->default;
  }

  /**
   * Returns a boolean value indicating whether a value is set to 'enabled'.
   */
  function isEnabled($field) {
    return isset($this->_data[$field]) ? $this->_data[$field] == 1 : FALSE;
  }

  /**
   * Returns enabled array values or false if the setting is not an array.
   */
  function getEnabled() {
    $enabled = array();
    foreach ($this->_data as $key => $value) {
      if ($this->isEnabled($key)) {
        $enabled[$key] = $key;
      }
    }
    return $enabled;
  }
}
