<?php

/**
 * @file
 * Defines link type classes for Autolink module.
 */

/**
 * Defines the link collector interface for collecting links for processing.
 */
interface AutolinkCollectorInterface {
  function getLinks();
}

/**
 * Defines the Autolink link collector class for loading link objects for processing.
 */
class AutolinkLinkCollector implements AutolinkCollectorInterface {
  private $node = NULL;
  private $_info = array();
  protected $_reset = FALSE;
  const CACHE_NAME = 'cache_autolink';

  function __construct($node) {
    $this->node = $node;
    $this->_info = autolink_get('PluginInfo')->getCurrent();
    // Check whether the cache should be reset for this node type.
    $reset = variable_get('autolink_plugin_reset', node_get_types('names'));
    if (!empty($reset[$this->node->type])) {
      $this->_reset = TRUE;
      unset($reset[$this->node->type]);
      variable_set('autolink_plugin_reset', $reset);print_r($reset);
    }
  }

  function __destruct() {
    $this->node = NULL;
    $this->_info = array();
  }

  /**
   * Gathers and returns an array of all available links based on enabled settings.
   */
  function getLinks() {
    return $this->getDefined() + $this->getAbstract();
  }

  /**
   * Decides whether to return links from cache or reset links.
   */
  protected function getDefined() {
    if ($this->_reset) {
      return $this->fetchDefined();
    }
    elseif ($cache = cache_get('defined_links_'. $this->node->type, self::CACHE_NAME)) {
      return $cache->data;
    }
    else {
      return $this->fetchDefined();
    }
    return $links;
  }

  /**
   * Returns an array of defined link objects. Link objects are cached
   * based on the current node type.
   */
  protected function fetchDefined() {
    $links = array();
    $available = array();
    foreach ($this->_info['link types'] as $type) {
      $available[] = "a.type = '$type'";
    }
    if ($available) {
      $result = db_query("SELECT a.* FROM {autolink_link} a INNER JOIN {autolink_group} g ON a.group_type = g.type LEFT JOIN {autolink_group_node_type} t ON a.group_type = t.group_type WHERE g.status <> 0 AND t.node_type = '%s' AND ". implode (' OR ', $available) ."", $this->node->type);
      while ($data = db_fetch_array($result)) {
        AutolinkLinkMapper::prepare($data);
        $links[] = autolink_get('link', $data);
      }
    }
    cache_set('defined_links_'. $this->node->type, $links, self::CACHE_NAME, CACHE_TEMPORARY);
    return $links;
  }

  /**
   * Decides whether to return links from cache or reset links.
   */
  protected function getAbstract() {
    if ($this->_reset) {
      return $this->fetchAbstract();
    }
    if ($cache = cache_get('abstract_links_'. $this->node->type, self::CACHE_NAME)) {
      return $cache->data;
    }
    else {
      return $this->fetchAbstract();
    }
  }

  /**
   * Retrieves abstract links from providers.
   *
   * Links are retrieved in an array of sub-arrays which should hold at least
   * keyword and path data. That data is converted into new AutolinkLink objects
   * using the link type class, so link type class methods are still available.
   * Link objects are then cached based on the current node type.
   */
  protected function fetchAbstract() {
    $links = array();
    $active = autolink_get('LinkTypeInfo')->getActive();
    $config = autolink_get('config')->fetchArraySetting('link_types');

    // Load only plugin supported link types.
    foreach ($active as $type => $info) {
      // Link types must be enabled and not definable or not identifiable.
      if (($info['definable'] == AUTOLINK_DISABLED && $config->isEnabled($type)) || $info['identifiable'] == AUTOLINK_DISABLED) {
        Autolink::includeFile($info['module'], $info['file']);
        $class = $info['provider'];
        if (class_exists($class)) {
          $object = new $class();
          foreach ($object->provideLinks($this->node) as $data) {
            $data['type'] = $type;
            $data['generic'] = TRUE;
            $links[] = autolink_get('link', $data);
          }
        }
        else {
          throw new AutolinkException(AutolinkException::ClassFailed);
        }
      }
    }
    cache_set('abstract_links_'. $this->node->type, $links, self::CACHE_NAME, CACHE_TEMPORARY);
    return $links;
  }
}

/**
 * Defines the link mapper interface for loading and saving link objects.
 */
interface AutolinkMapperInterface {
  function save(array $data);
  function load($lid);
  function delete($lid);
}

/**
 * Defines the link mapper class, which saves and loads data from the database.
 */
class AutolinkLinkMapper implements AutolinkMapperInterface {

  function __construct() {}

  /**
   * Saves link data to the database.
   */
  function save(array $data) {
    if (!empty($data['lid']) && !empty($data['keyword']) && !empty($data['path'])) {
      module_invoke_all('autolink_update', (object) $data);
      $this->presave($data);
      drupal_write_record('autolink_link', $data, 'lid');
      $status = SAVED_UPDATED;
    }
    elseif (!empty($data['lid'])) {
      $status = $this->delete($data['lid']);
    }
    elseif (!empty($data['type'])) {
      module_invoke_all('autolink_insert', (object) $data);
      $data['keyword'] = check_plain(trim($data['keyword']));
      $this->presave($data);
      drupal_write_record('autolink_link', $data);
      $status = SAVED_NEW;
    }
    // Clear the Autolink cache to ensure that the link is accessible.
    cache_clear_all(NULL, 'cache_autolink');
    return $status;
  }

  /**
   * Prepares data to be saved by storing extra data in serialized array.
   */
  private function presave(array &$data) {
    // Convert the group back to a format appropriate for drupal_write_record().
    // This is done because the field 'group' is not allowed in the db table.
    $data['group_type'] = isset($data['group']) ? $data['group'] : $data['group_type'];
    unset($data['group']);
    // Store values in 'extra' if they are not standard values.
    if (!isset($data['extra'])) {
      $defaults = array('lid', 'type', 'group_type', 'keyword', 'path', 'query', 'method');
      $data['extra'] = array();
      foreach ($data as $key => $value) {
        if (!in_array($key, $defaults)) {
          $data['extra'][$key] = $value;
          unset($data[$key]);
        }
      }
    }
    $data['extra'] = serialize($data['extra']);
  }

  /**
   * Returns an array of link data.
   */
  function load($lid) {
    try {
      $data = self::fetchData($lid);
      self::prepare($data);
      return $data;
    }
    catch (AutolinkException $e) {
      $e->displayMessage();
    }
  }

  /**
   * Queries the database to retrieve link data.
   */
  protected static function fetchData($lid) {
    if ($data = db_fetch_array(db_query('SELECT * FROM {autolink_link} WHERE lid = %d', $lid))) {
      $result = db_query("SELECT node_type FROM {autolink_group_node_type} WHERE group_type = '%s'", $data['group_type']);
      $data['group'] = $data['group_type'];
      unset($data['group_type']);
      $data['node_types'] = array();
      while ($type = db_result($result)) {
        $data['node_types'][] = $type;
      }
      return $data;
    }
    else {
      throw new AutolinkException(AutolinkException::DatabaseFetchFailed);
    }
  }

  /**
   * Prepares data from the database for use in link type classes.
   */
  public static function prepare(array &$data) {
    $data['extra'] = !empty($data['extra']) ? unserialize($data['extra']) : array();
    if (!empty($data['extra']) && is_array($data['extra'])) {
      foreach ($data['extra'] as $key => $value) {
        $data[$key] = $value;
      }
    }
    unset($data['extra']);
  }

  /**
   * Deletes a link.
   */
  function delete($lid) {
    module_invoke_all('autolink_delete', autolink_load($lid));
    db_query('DELETE FROM {autolink_link} WHERE lid = %d', $lid);
  }
}

/**
 * Defines the link interface.
 */
interface AutolinkLinkType {
  function save();
  function get($property);
  function getType();
  function getGroup();
  function getKeyword();
  function getPath();
  function getQuery();
  function getInfo();
}

/**
 * Defines the Autolink link type class.
 */
class AutolinkLink implements AutolinkLinkType {
  private $_data = array();
  public $type = NULL;
  public $_info = NULL;
  protected $config;

  /**
   * Standard constructor method for link objects.
   *
   * @param $data
   *   An array of data with which to populate the object's data property.
   * @param $type
   *   The link type of the link object being created.
   * @param $info
   *   An array of link type info as defined in hook_autolink_link_type_info().
   */
  function __construct(array $data, $type, $info = array()) {
    $this->type = $type;
    $this->_info = !empty($info) ? $info : autolink_get('LinkTypeInfo')->getLinkTypeInfo($type);
    $this->_data = $data;
    $this->config = autolink_get('config');
  }

  function __destruct() {
    $this->type = NULL;
    $this->_data = array();
    $this->_info = NULL;
  }

  function __get($property) {
    return $this->$property;
  }

  function __set($property, $value) {
    $this->$property = $value;
  }

  /**
   * Saves the link object to the database.
   */
  function save() {
    // Restore the link type to the array of data to save.
    $data = $this->_data;
    $data['type'] = isset($data['type']) ? $data['type'] : $this->getType();
    autolink_get('LinkMapper')->save($data);
  }

  /**
   * Returns a specific data property based on the given string parameter.
   */
  function get($property) {
    return isset($this->_data[$property]) ? $this->_data[$property] : FALSE;
  }

  /**
   * Returns the link type info array.
   */
  function getInfo() {
    return $this->_info;
  }

  /**
   * Returns the link ID.
   */
  function getIdentifier() {
    return isset($this->_data['lid']) ? $this->_data['lid'] : FALSE;
  }

  /**
   * Returns the link type of the object.
   */
  function getType() {
    return $this->type;
  }

  /**
   * Returns the machine-name of the link group.
   */
  function getGroup() {
    if (!isset($this->_data['group_type']) && isset($this->_data['group'])) {
      return $this->_data['group'];
    }
    return $this->_data['group_type'];
  }

  /**
   * Returns the link keyword.
   */
  function getKeyword() {
    return $this->_data['keyword'];
  }

  /**
   * Returns the link path.
   */
  function getPath() {
    return $this->_data['path'];
  }

  /**
   * Returns the link query.
   */
  function getQuery() {
    return $this->_data['query'];
  }

  /**
   * The following methods are for example only.
   */

  /**
   * The setAttributes function may be defined by link types. If the plugin
   * 'attributes' setting is not set to AUTOLINK_ABSOLUTE then the link type
   * attributes method will be used if found.
   */
  //function getAttributes() {
  //  return array(
  //     'title' => $this->get('keyword'),
  //  );
  //}

  /**
   * The setText method is used to get the text of a link. The same properties
   * as discussed above apply here.
   */
  //function getText($match) {
  //  return $match;
  //}

}

/**
 * Defines the link provider class for abstract link types.
 */
abstract class AutolinkProvider {
  protected $config;

  function __construct() {
    $this->config = autolink_get('config');
  }

  abstract function provideLinks($type);
}
