<?php

/**
 * @file
 * Provides link type classes for the Autolink Default plugin.
 */

/**
 * Defines the standard link class for defining regular expressions.
 */
class AutolinkDefaultLink extends AutolinkLink {

  /**
   * Returns attributes based on the current link.
   */
  protected function attributes() {
    return array(
      'title' => $this->get('keyword'),
      'style' => $this->config->get('autolink_style', ''),
      'class' => $this->config->get('autolink_class', ''),
      'target' => menu_path_is_external($this->get('path')) ? $this->config->get('external_target', 'none') : $this->config->get('internal_target', 'none'),
      'rel' => $this->config->fetchSetting('external_rel', 1)->isEnabled() && menu_path_is_external($this->get('path')) ? 'nofollow' : '',
    );
  }

  /**
   * Returns a boolean value indicating whether or not the link is case sensitive.
   */
  function getCase() {
    return $this->get('case') == 1;
  }

  /**
   * Returns an array of stored synonyms.
   */
  function getSynonyms() {
    return $this->get('synonyms');
  }

}

/**
 * Defines the node link processor class.
 */
class AutolinkNodeLink extends AutolinkDefaultLink {

  /**
   * Builds a token title using the link to node if a pattern has been defined.
   */
  function getAttributes() {
    $parts = explode('/', $this->get('path'));
    if (module_exists('token')) {
      $node = node_load($parts[1]);
      if ($node) {
        $attributes = $this->attributes();
        $pattern = $this->config->get('node_title', '[title]');
        $attributes['title'] = !empty($pattern) && module_exists('token') ? token_replace($pattern, 'node', $node) : $this->get('keyword');
        return $attributes;
      }
    }
    return $this->attributes();
  }

}

/**
 * Defines the user link processor class.
 */
class AutolinkUserLink extends AutolinkDefaultLink {

  /**
   * Builds a Token title using the user to which the link links.
   */
  function getAttributes() {
    $parts = explode('/', $this->get('path'));
    if ($user = user_load(array('uid' => $parts[1])) && module_exists('token')) {
      $attributes = $this->attributes();
      $pattern = $this->config->get('user_title', '[user]');
      $attributes['title'] = !empty($pattern) && module_exists('token') ? token_replace($pattern, 'user', $user) : $this->get('keyword');
      return $attributes;
    }
    return $this->attributes();
  }

}

/**
 * Defines the taxonomy term link processor class.
 */
class AutolinkTermLink extends AutolinkDefaultLink {

  /**
   * Returns attributes with a token based title.
   */
  function getAttributes() {
    $parts = explode('/', $this->get('path'));
    if ($term = $this->termLoad($parts[2]) && module_exists('token')) {
      $attributes = $this->attributes();
      $pattern =  $this->config->get('term_title', '[cat]');
      $attributes['title'] = !empty($pattern) && module_exists('token') ? token_replace($pattern, 'term', $term) : $this->get('keyword');
      return $attributes;
    }
    return $this->attributes();
  }

  /**
   * Loads a taxonomy term for use in token replacements.
   */
  private function termLoad($tid) {
    return db_fetch_object(db_query('SELECT * FROM {term_data} WHERE tid = %d', $tid));
  }

}
