<?php

/**
 * @file
 * Defines the Autolink error handler class.
 */

/**
 * Defines the error messaging interface.
 */
interface AutolinkErrorInterface {
  public function addMessage($message);
  public function addMultiple(array $info);
  public function displayAll();
  public function recordAll();
  public function clear();
}

/**
 * Displays error messages to the screen.
 */
class AutolinkErrorHandler implements AutolinkErrorInterface {
  private static $instance;
  public $messages = array();
  protected $flag = NULL;
  const ERROR_FILE = '/error.php';

  function __construct() {
    $this->flag = 'error';
  }

  function __destruct() {
    $this->messages = array();
    $this->flag = NULL;
  }

  function __set($property, $value) {
    $method = "set{$property}";
    if (method_exists($this, $method)) {
      $this->$method($value);
    }
  }

  function __toString() {
    return implode("\n", $this->messages);
  }

  /**
   * Sets the message flag.
   */
  function setFlag($flag) {
    $this->flag = $flag;
  }

  /**
   * Adds an error message to the messages array for display.
   */
  public function addMessage($message) {
    $this->messages[] = $message;
  }

  /**
   * Adds an error message and extra information to the messages array.
   */
  public function addMultiple(array $info) {
    $this->messages[] = implode("\n", $info);
  }

  /**
   * Displays error messages on the screen with drupal_set_message().
   */
  public function displayAll() {
    if (!empty($this->messages)) {
      foreach ($this->messages as $message) {
        drupal_set_message($message, $this->flag);
      }
      // Clear the messages to reset the static object.
      $this->clear();
    }
  }

  /**
   * Records a message to the error log.
   */
  public function recordAll() {
    if (!empty($this->messages)) {
      foreach ($this->messages as $message) {
        file_put_contents(self::ERROR_FILE, $this->__toString(), FILE_APPEND);
      }
      // Clear the messages to reset the static object.
      $this->clear();
    }
  }

  /**
   * Clears the contents of the error message handler.
   */
  public function clear() {
    $this->messages = array();
  }
}

/**
 * Defines the primary error handling class.
 */
class AutolinkException extends Exception {
  protected $_mode = NULL;
  const PRODUCTION_MODE = 0;
  const DEBUG_MODE = 1;
  const IncludeFailed = 1;
  const ClassFailed = 2;
  const FormFailed = 3;
  const PluginInfo = 4;
  const LinkTypeInfo = 5;
  const Plugin = 6;
  const LinkType = 7;
  const LibraryFailed = 8;
  const DatabaseFetchFailed = 9;
  const MissingField = 10;

  /**
   * Sets the message based on the error code given to the exception.
   * This allows for central management of messages and printing of custom
   * messages on errors with drupal_set_message().
   *
   * @param $code
   *   An error code, used to determine which message to set on the error.
   * @params arguments
   *   A differing number of arguments that must be used with certain error messages.
   *   Arguments are retrieved via func_get_args(), so this depends on the error.
   */
  function __construct() {
    $args = func_get_args();
    $code = $args[0];
    $this->_mode = variable_get('autolink_development_mode', 0);
    if ($this->_mode == 1) {
      switch ($code) {
        case AutolinkException::IncludeFailed:
          $message = t('File include failed: Cannot locate file at %path.', array('%path' => $args[1]));
          break;
        case AutolinkException::ClassFailed:
          $message = t('Class instantiation failed: cannot locate %class class.', array('%class' => $args[1]));
          break;
        case self::LibraryFailed:
          $message = t('File include failed: Cannot locate the Simple HTML DOM library.');
          break;
        case AutolinkException::FormFailed:
          $message = t('Form building failed: cannot locate %method method for form %form defined by link type %link_type.', array('%method' => $args[1], '%form' => $args[2], '%link_type' => $args[3]));
          break;
        case AutolinkException::PluginInfo:
          $message = t('Plugin information misconfigured: %message', array('%message' => $args[1]));
          break;
        case AutolinkException::LinkTypeInfo:
          $message = t('Link type information misconfigured: %message', array('%message' => $args[1]));
          break;
        case AutolinkException::Plugin:
          $message = t('Plugin failed: %message', array('%message' => $args[1]));
          break;
        case AutolinkException::LinkType:
          $message = t('Link type failed: %message', array('%message' => $args[1]));
          break;
        case AutolinkException::DatabaseFetchFailed:
          $message = t('Data fetch failed: Cannot locate link data.');
          break;
        case AutolinkException::MissingField:
          $message = t('Node processing failed: Cannot locate field %field.', array('%field' => $args[1]));
          break;
      }
      parent::__construct($message, $code);
    }
  }

  public function setMode($mode) {
    $this->_mode = $mode;
  }

  public function getMode() {
    return $this->_mode;
  }

  /**
   * Logs an error message to the Autolink error messager for display.
   */
  public function logMessage(AutolinkErrorHandler $handler) {
    switch ($this->_mode) {
      case self::PRODUCTION_MODE:
        $handler->addMessage($this->getMessage());
        break;
      case self::DEBUG_MODE:
        $this->logMultiple($handler, array('message', 'file', 'line', 'TraceAsString'));
        break;
      case 'default':
        $handler->addMessage($this->getMessage());
        break;
    }
  }

  /**
   * Logs an error message and additional information to the error messager.
   *
   * @param $messager
   *   The AutolinkErrorMessager object being used to display messages.
   * @param $info
   *   An array of values indicating which information to display in the
   *   error message. These values directly relate to Exception class
   *   methods like getLine() or getFile().
   */
  function logMultiple(AutolinkErrorHandler $handler, array $info) {
    $values = array();
    foreach ($info as $type) {
      // Ensure that the first letter is capitalized.
      $property = drupal_ucfirst($type);
      $method = "get{$property}";
      $values[] = $property .': '. $this->$method();
    }
    $handler->addMultiple($values);
  }

  /**
   * Prints an individual error message to the screen.
   *
   * @param $flag
   *   A string indicating the level of warning to pass to drupal_set_message().
   * @param $debug
   *   A boolean value indicating whether this message may be displayed in
   *   debug mode. This defaults to TRUE, but displaying errors in debug mode
   *   still require $_mode to be set to DEBUG. Set $debug to FALSE to prevent
   *   debug information from being displayed where it is not necessary.
   */
  function displayMessage($flag = 'error', $debug = TRUE) {
    switch ($this->_mode) {
      case self::PRODUCTION_MODE:
        drupal_set_message($this->getMessage(), $flag);
        break;
      case self::DEBUG_MODE:
        if ($debug) {
          drupal_set_message(implode("\n", array($this->getMessage(), $this->getFile(), $this->getLine(), $this->getTraceAsString())), $flag);
        }
        else {
          drupal_set_message($this->getMessage(), $flag);
        }
        break;
    }
  }
}
