<?php

/**
 * @file
 * Provides the Default Autolink Plugin for the Autolink package.
 */

/**
 * Autolink plugin class.
 */
class AutolinkDefaultPlugin extends AutolinkPlugin {

  /**
   * Returns an array of fields to process.
   */
  function getFields($type) {
    return array('body', 'teaser');
  }

  /**
   * Returns an array of keyword synonyms.
   */
  function getSynonyms(AutolinkLink $link) {
    return $link->getSynonyms();
  }

  /**
   * Sets a limit on link generation per node.
   */
  function getTotalLimit() {
    return $this->config->get('total_link_limit', AUTOLINK_DEFAULT_LIMIT);
  }

  /**
   * Sets a limit on a per-link per-node basis.
   *
   * This method is set to ABSOLUTE in plugin info, so it cannot be
   * overridden without first altering plugin information.
   */
  function getLinkLimit() {
    return $this->config->get('link_limit', AUTOLINK_DEFAULT_LIMIT);
  }

  /**
   * Returns an array to prepend and append to the Autolink regular expression.
   */
  function getExpression(AutolinkLink $link) {
    return array('prepend' => '\b', 'append' => '\b');
  }

  /**
   * Returns the text of all links for the plugin. This method is set
   * to ABSOLUTE, so it cannot be overridden without altering plugin info.
   */
  function getText(AutolinkLink $link, $match) {
    return $match;
  }

  /**
   * Returns attributes for a link.
   *
   * This method is set to ENABLED in plugin info, so it may be overridden
   * by link types.
   */
  function getAttributes(AutolinkLink $link) {
    // Call the attributes method in the link type if it exists.
    // This is done to allow node, user, and term links to build token titles.
    if (method_exists($link, 'getAttributes')) {
      return $link->getAttributes();
    }

    return array(
      'title' => $link->get('keyword'),
      'style' => $this->config->get('autolink_style', ''),
      'class' => $this->config->get('autolink_class', ''),
      'target' => menu_path_is_external($link->get('path')) ? $this->config->get('external_target', 'none') : $this->config->get('internal_target', 'none'),
      'rel' => $this->config->fetchSetting('external_rel', 1)->isEnabled() && menu_path_is_external($link->get('path')) ? 'nofollow' : '',
    );
  }

  /**
   * Returns a bookean value representing the case sensitivity of a link.
   */
  function getCase(AutolinkLink $link) {
    if (method_exists($link, 'getCase')) {
      return $link->getCase();
    }
    return FALSE;
  }
}
