<?php

/**
 * @file
 * Administrative forms for autolink_ module.
 */

/**
 * Settings form for autolink.
 *
 * @ingroup forms
 * @see autolink_form_settings_submit()
 */
function autolink_form_settings(&$form_state, $edit = array()) {
  return autolink_get('config')->setForm()->execute('form', $form_state, $edit);
}

/**
 * Validation function for settings form.
 */
function autolink_form_settings_validate($form, &$form_state) {
  return autolink_get('config')->setForm()->execute('validate', $form, $form_state);
}

/**
 * Submit handler for Autolink settings form.
 *
 * This handler sets miltiple value field values by creating individual variables
 * for each value. This makes node and link type settings much easier to work with
 * on other forms and in processing functions and is cleaner than using the
 * 'array_filter' field.
 */
function autolink_form_settings_submit($form, &$form_state) {
  autolink_get('config')->setForm()->execute('submit', $form, $form_state);
}

/**
 * Callback for generating the links overview form.
 */
function autolink_admin_links($list) {
  Autolink::loadFiles(array('filters', 'operations'));
  if (!empty($_POST['links']) && isset($_POST['operation']) && ($_POST['operation'] == 'delete')) {
    $output = drupal_get_form('autolink_link_multiple_delete_confirm');
  }
  else {
    drupal_set_title(t('Autolink links'));
    $output = drupal_get_form('autolink_overview_links');
  }
  return $output;
}

/**
 * Form builder for the group/links overview page.
 *
 * @see autolink_admin_links()
 * @see autolink_overview_links_validate()
 * @see autolink_overview_links_submit()
 */
function autolink_overview_links($form_state = array()) {
  drupal_add_css(drupal_get_path('module', 'autolink') .'/autolink.css');
  $links = array();
  $result = db_query('SELECT * FROM {autolink_link}');
  while ($link = db_fetch_object($result)) {
    $links[$link->lid] = $link;
  }

  $form['options'] = array(
    '#type' => 'fieldset',
    '#title' => t('Update links'),
    '#prefix' => '<div class="container-inline">',
    '#suffix' => '</div>',
  );

  // Add our own operations first.
  $options = array();
  $ops = autolink_operations();
  foreach ($ops as $operation => $value) {
    $options[$operation] = $value['label'];
  }

  // Invoke hook_autolink_operations() to get a list of external operations.
  foreach (module_invoke_all('autolink_operations') as $operation => $value) {
    $options[$operation] = $value['label'];
  }

  $form['options']['operation'] = array(
    '#type' => 'select',
    '#options' => $options,
    '#default_value' => 'activate',
  );
  $form['options']['submit'] = array(
    '#type' => 'submit',
    '#value' => t('Update'),
  );

  $form['data'] = array('#type' => 'value', '#value' => $links);

  foreach ($links as $link) {
    $checkboxes[$link->lid] = '';
    $form['type'][$link->lid] = array('#value' => $link->type);
    $form['keyword'][$link->lid] = array('#value' => $link->keyword);
    $form['destination'][$link->lid] = array('#value' => drupal_lookup_path('alias', $link->path) ? drupal_lookup_path('alias', $link->path) : $link->path);
    $form['operations'][$link->lid]['edit'] = array('#value' => l(t('edit'), "admin/content/autolink/$link->group_type/edit/link/$link->lid", array('query' => "admin/content/autolink")));
    $options[$link->lid] = '';
  }

  $form['links'] = array(
    '#type' => 'checkboxes',
    '#options' => $checkboxes,
  );

  return $form;
}

/**
 * Theme callback for the links overview form.
 *
 * This callback separates links by group. For each link group a
 * fieldset is created with a description and applied content types.
 * The form also has bulk operations that are defined through
 * hook_autolink_link_operations().
 *
 * @see autolink_overview_links()
 * @see hook_autolink_link_operations()
 * @ingroup themeable
 */
function theme_autolink_overview_links($form) {
  if (isset($form['confirm'])) {
    return drupal_render($form);
  }

  // Retrieve link group information for populating fieldsets.
  $groups = array();
  $group_result = db_query('SELECT g.*, t.node_type FROM {autolink_group} g LEFT JOIN {autolink_group_node_type} t ON g.type = t.group_type');
  while ($group = db_fetch_object($group_result)) {
    $types = array();
    $types_result = db_query("SELECT t.* FROM {autolink_group_node_type} g LEFT JOIN {node_type} t ON g.node_type = t.type WHERE g.group_type = '%s'", $group->type);
    while ($type = db_fetch_object($types_result)) {
      $types[$type->type] = $type->name;
    }
    $group->node_types = $types;
    $groups[$group->type] = $group;
  }

  $header = array(
    theme('tabel_select_header_cell'),
    array('data' => t('Type')),
    array('data' => t('Keyword')),
    array('data' => t('Destination')),
    t('Operations'),
  );

  // Populate each group with its links.
  $sections = array();
  $links = $form['data']['#value'];
  foreach ($links as $link) {
    $sections[$link->group_type][$link->lid] = $link;
  }

  // Ensure that even empty groups get displayed to show group links.
  foreach ($groups as $group) {
    if (!isset($sections[$group->type])) {
      $sections[$group->type] = array();
    }
  }
  ksort($sections);

  // Render the bulk operations form and separate links into groups.
  $output = drupal_render($form['options']);
  foreach ($sections as $group => $links) {
    $row = array();
    if (!empty($links)) {
      foreach ($links as $key => $value) {
        $rows[] = array(
          drupal_render($form['links'][$key]),
          drupal_render($form['type'][$key]),
          drupal_render($form['keyword'][$key]),
          drupal_render($form['destination'][$key]),
          drupal_render($form['operations'][$key]),
        );
      }
      $value = theme('table', $header, $rows, array('class' => 'autolink-group'));
    }
    else {
      $value = '';
    }

    // Create a fieldset with the group description, node types, and links.
    $description = '<div class="autolink-group-info">';
    $description .= '<div class="autolink-group-description">' .  $groups[$group]->description . '</div>';
    $description .= '<div class="autolink-group-node-types">' . implode(', ', $groups[$group]->node_types) . '</div>';
    $description .= '<div class="autolink-group-links">';
    $description .= '<ul><li>' . l(t('edit group'), 'admin/content/autolink/edit/group/' . $groups[$group]->type, array('query' => 'admin/content/autolink')) . '</li>';
    $description .= '<li>' . l(t('add link'), 'admin/content/autolink/' . $groups[$group]->type . '/add/link', array('query' => 'admin/content/autolink')) . '</li></ul>';
    $description .= '</div></div>';

    $fieldset = array(
      '#title' => t($groups[$group]->name),
      '#description' => $description,
      '#collapsible' => TRUE,
      '#value' => $value,
    );
    $output .= theme('fieldset', $fieldset);
  }

  $output .= drupal_render($form);
  return $output;
}

/**
 * Form validation for autolink_ groups administration update form.
 *
 * @see autolink_overview_links()
 * @see autolink_overview_links_submit()
 */ 
function autolink_overview_links_validate($form, &$form_state) {
  $form_state['values']['links'] = array_filter($form_state['values']['links']);
  if (!count($form_state['values']['links'])) {
    form_set_error('', t('No links were selected.'));
  }
}

/**
 * Submit handler for autolink_ administration update form.
 *
 * @see autolink_overview_links()
 * @see autolink_overview_links_validate()
 */
function autolink_overview_links_submit($form, &$form_state) {
  $operations = autolink_operations($form_state);
  $operations += module_invoke_all('autolink_operations', $form_state);
  $operation = $operations[$form_state['values']['operation']];
  // Filter out unchecked links.
  $links = array_filter($form_state['values']['links']);
  if ($function = $operation['callback']) {
    // Add in callback arguments if present.
    if (isset($operation['callback arguments'])) {
      $args = array_merge(array($links), $operation['callback arguments']);
    }
    else {
      $args = array($links);
    }
    // Call the function based on the operation that was selected.
    call_user_func_array($function, $args);
    drupal_set_message(t('The update has been performed.'));
  }
}

/**
 * Display form for adding and editing link groups.
 *
 * @ingroup forms
 * @see autolink_form_group_submit()
 */
function autolink_form_group(&$form_state, $edit = array()) {
  $form['identification'] = array(
    '#type' => 'fieldset',
    '#title' => t('Identification'),
    '#collapsible' => TRUE,
  );
  $form['identification']['status'] = array(
    '#type' => 'select',
    '#title' => t('Status'),
    '#default_value' => isset($edit['status']) ? $edit['status'] : 1,
    '#options' => array(0 => t('Disabled'), 1 => t('Active')),
  );
  $form['identification']['name'] = array(
    '#type' => 'textfield',
    '#title' => t('Group name'),
    '#default_value' => isset($edit['name']) ? $edit['name'] : '',
    '#maxlength' => 255,
    '#description' => t('The human-readable name for this link group.'),
    '#required' => TRUE,
  );
  $form['identification']['description'] = array(
    '#type' => 'textarea',
    '#title' => t('Description'),
    '#default_value' => isset($edit['description']) ? $edit['description'] : '',
    '#description' => t('Description of the link group.'),
  );
  $form['nodes'] = array(
    '#type' => 'fieldset',
    '#title' => t('Content types'),
    '#collapsible' => TRUE,
  );
  $form['nodes']['node_types'] = array(
    '#type' => 'checkboxes',
    '#title' => t('Content types'),
    '#default_value' => isset($edit['types']) ? $edit['types'] : array(),
    '#options' => array_map('check_plain', node_get_types('names')),
    '#description' => t('Select content types for this link group to monitor.'),
  );

  $form['submit'] = array('#type' => 'submit', '#value' => t('Save'));
  if (isset($edit['type'])) {
    $form['delete'] = array('#type' => 'submit', '#value' => t('Delete'));
    $form['type'] = array('#type' => 'value', '#value' => $edit['type']);
  }
  else {
    // Add a validate function to ensure group names are not duplicated.
    $form['#valdate'][] = 'autolink_form_group_duplicate_validate';
  }

  return $form;
}

/**
 * Validate function for add link group form.
 */
function autolink_form_group_duplicate_validate($form, &$form_state) {
  $group = $form_state['values']['name'];
  $result = db_query("SELECT * FROM {autolink_group} WHERE name = '%s'", $group);
  if (db_result($result)) {
    form_set_error('name', t('A link group with the name %name already exists.', array('%name' => $group)));
  }
}

/**
 * Submit function for add link group form.
 */
function autolink_form_group_submit($form, &$form_state) {
  // Fix up the nodes array to remove unchecked nodes.
  $form_state['values']['types'] = array_filter($form_state['values']['node_types']);
  switch (autolink_group_save($form_state['values'])) {
    case SAVED_NEW:
      drupal_set_message(t('Created new link group %name.', array('%name' => $form_state['values']['name'])));
      watchdog('autolink', 'Created new link group %name.', array('%name' => $form_state['values']['name']), WATCHDOG_NOTICE, l(t('edit'), 'admin/content/autolink/edit/group/'. $form_state['values']['type']));
      break;
    case SAVED_UPDATED:
      drupal_set_message(t('Updated link group %name.', array('%name' => $form_state['values']['name'])));
      watchdog('autolink', 'Updated link group %name.', array('%name' => $form_state['values']['name']), WATCHDOG_NOTICE, l(t('edit'), 'admin/content/autolink/edit/group/'. $form_state['values']['type']));
      break;
  }

  $form_state['type'] = $form_state['values']['type'];
  $form_state['redirect'] = 'admin/content/autolink';
  return;
}

/**
 * Page to edit a link group.
 */
function autolink_admin_group_edit($group) {
  if ((isset($_POST['op']) && $_POST['op'] == t('Delete')) || isset($_POST['confirm'])) {
    return drupal_get_form('autolink_group_confirm_delete', $group->type);
  }
  return drupal_get_form('autolink_form_group', (array)$group);
}

/**
 * Form builder for the link group delete confirmation form.
 *
 * @ingroup forms
 * @see autolink_group_confirm_delete_submit()
 */
function autolink_group_confirm_delete(&$form_state, $type) {
  $group = autolink_group_load($type);

  $form['type'] = array('#type' => 'value', '#value' => $type);
  $form['name'] = array('#type' => 'value', '#value' => $group->name);
  return confirm_form($form,
                  t('Are you sure you want to delete the link group %name?',
                  array('%name' => $group->name)),
                  'admin/content/autolink',
                  t('Deleting a link group will delete all the link definitions within it. This action cannot be undone.'),
                  t('Delete'),
                  t('Cancel'));
}

/**
 * Submit handler to delete a link group after confirmation.
 *
 * @see autolink_group_confirm_delete()
 */
function autolink_group_confirm_delete_submit($form, &$form_state) {
  $status = autolink_group_delete($form_state['values']['type']);
  drupal_set_message(t('Deleted link group %name.', array('%name' => $form_state['values']['name'])));
  watchdog('autolink', 'Deleted link group %name.', array('%name' => $form_state['values']['name']), WATCHDOG_NOTICE);
  $form_state['redirect'] = 'admin/content/autolink';
  return;
}

/**
 * Form for adding or editing a link definition.
 *
 * This function uses the AHAH Helper module to build a dynamic link definition
 * form. When the AHAH submit function is called and the form is rebuilt this
 * function builds link type specific form elements based on data retrieved with
 * the link type that has been selected.
 */
function autolink_form_link(&$form_state, $group, $settings = array()) {
  Autolink::loadFiles(array('config', 'forms'));
  $info = autolink_get('LinkTypeInfo');
  $config = autolink_get('Config');

  $form = array();
  $form['group'] = array('#type' => 'value', '#value' => $group->type);

  ahah_helper_register($form, $form_state);

  // Prepare link types for the form.
  $select_type = array('select' => t('--Select a link type--'));
  $link_types = $config->getSettings('link_types', $info->getDefinable('label'), 1);
  asort($link_types);

  if (!empty($link_types)) {
    $link_type_options = array_merge($select_type, $link_types);
  }
  else {
    $link_type_options = $select_type;
    drupal_set_message(t('There are currently no link types enabled. Link types may be enabled by visiting the setting page.'), 'warning');
  }

  if (!isset($form_state['storage']['link_wrapper']['type'])) {
    $link_type = isset($settings['type']) ? $settings['type'] : 'select';
  }
  else {
    $link_type = $form_state['storage']['link_wrapper']['type'];
  }

  $form['link_wrapper'] = array(
    '#prefix' => '<div id="link-wrapper">',
    '#suffix' => '</div>',
    '#tree'   => TRUE,
  );
  // If there is only one link type then set that as the default instead of 'select'.
  $form['link_wrapper']['type'] = array(
    '#type' => 'select',
    '#title' => t('Link to'),
    '#options' => $link_type_options,
    '#default_value' => $link_type,
    '#ahah' => array(
      'event' => 'change',
      'path' => ahah_helper_path(array('link_wrapper')),
      'wrapper' => 'link-wrapper',
      'effect' => 'fade',
    ),
  );
  $form['link_wrapper']['type_submit'] = array(
    '#type' => 'submit',
    '#value' => t('Set link type'),
    '#submit' => array('ahah_helper_generic_submit'),
    '#attributes' => array('class' => 'no-js'),
  );

  // Only if a link type has been selected do we show the link definition form.
  if (!empty($link_type) && $link_type != 'select') {
    $form['link_wrapper']['definition'] = array(
      '#type' => 'fieldset',
      '#title' => t('Definition'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
    );
    $link_type_info = $info->getLinkTypeInfo($link_type, 'config');

    // Display an additional selector if there are multiple forms for the link type.
    if (count($link_type_info) > 1) {
      // Set the method for edit forms or from the form storage.
      if (!isset($form_state['storage']['link_wrapper']['definition']['method_wrapper']['method'])) {
        $selected_method = isset($settings['method']) ? $settings['method'] : 'no_method';
      }
      else {
        $selected_method = $form_state['storage']['link_wrapper']['definition']['method_wrapper']['method'];
      }

      $method_options['no_method'] = t('--Select a method--');
      $options = array();
      foreach ($info->getLinkTypeInfo($link_type, 'config') as $key => $value) {
        $options[$key] = $value['label'];
      }
      $method_options += $options;

      $form['link_wrapper']['definition']['method_wrapper'] = array(
        '#prefix' => '<div id="method-wrapper">',
        '#suffix' => '</div>',
        '#tree' => TRUE,
      );
      $form['link_wrapper']['definition']['method_wrapper']['method'] = array(
        '#type' => 'select',
        '#title' => t('Selection method'),
        '#options' => $method_options,
        '#default_value' => $selected_method,
        '#required' => TRUE,
        '#ahah' => array(
          'event' => 'change',
          'path' => ahah_helper_path(array('link_wrapper', 'definition', 'method_wrapper')),
          'wrapper' => 'method-wrapper',
          'effect' => 'fade',
        ),
      );

      if ($selected_method != 'no_method') {
        $type_form = AutolinkFormFactory::getInstance($link_type, $selected_method);
        $form['link_wrapper']['definition']['method_wrapper'] += $type_form->execute('form', $settings);
      }
    }
    else {
      $selected_method = key($link_type_info);
      $form['method'] = array('#type' => 'value', '#value' => $selected_method);
      $type_form = AutolinkFormFactory::getInstance($link_type, $selected_method);
      $form['link_wrapper']['definition'] += $type_form->execute('form', $settings);
    }
  }

  $form['submit'] = array('#type' => 'submit', '#value' => t('Save'));
  $form['group_type'] = array('#type' => 'value', '#value' => $group->type);
  if (isset($settings['lid'])) {
    $form['delete'] = array('#type' => 'submit', '#value' => t('Delete'));
    $form['lid'] = array('#type' => 'value', '#value' => $settings['lid']);
  }
  return $form;
}

/**
 * Validation handler for the link edit form.
 *
 * Here we ensure the link does not contain inappropriate characters. Also, since
 * we are using disabled and hidden fields, we have to do extra form validation
 * here because we cannot use the 'required' => TRUE property on a disabled field.
 *
 * @see autolink_form_link()
 */
function autolink_form_link_validate($form, &$form_state) {
  Autolink::loadFiles(array('forms'));
  $values = _autolink_merge_link_form_values($form_state['values']);
  AutolinkFormFactory::getInstance($values['type'], $values['method'])->execute('validate', $values);
}

/**
 * Final submit function for the link form.
 */
function autolink_form_link_submit($form, &$form_state) {
  Autolink::loadFiles(array('forms'));
  $values = _autolink_merge_link_form_values($form_state['values']);
  $object = AutolinkFormFactory::getInstance($values['type'], $values['method']);

  switch ($object->execute('save', $values)) {
    case SAVED_NEW:
      drupal_set_message(t('Created new <i>%link_type</i> type link for keyword <i>%keyword</i>.', array('%link_type' => $values['type'], '%keyword' => $values['keyword'])));
      watchdog('autolink', 'Created new <i>%link_type</i> type link for keyword <i>%keyword</i>.', array('%link_type' => $values['type'], '%keyword' => $values['keyword']), WATCHDOG_NOTICE, l(t('edit'), 'admin/content/autolink'));
      break;
    case SAVED_UPDATED:
      drupal_set_message(t('Updated <i>%link_type</i> type link for keyword <i>%keyword</i>.', array('%link_type' => $values['type'], '%keyword' => $values['keyword'])));
      watchdog('autolink', 'Updated link for keyword <i>%keyword</i>.', array('%keyword' => $values['keyword']), WATCHDOG_NOTICE, l(t('edit'), 'admin/content/autolink'));
      break;
  }

  $form_state['lid'] = $form_state['values']['lid'];
  drupal_goto('admin/content/autolink');

  return;
}

/**
 * Page to add a link.
 */
function autolink_add_link_page($group) {
  drupal_set_title(t('Add link definition to group %group', array('%group' => $group->name)));
  return drupal_get_form('autolink_form_link', $group);
}

/**
 * Page to edit a link.
 *
 * @param $link
 *   A link object.
 *
 * @see autolink_form_link()
 * @see autolink_confirm_delete()
 */
function autolink_admin_link_edit($group, $lid) {
  if ((isset($_POST['op']) && $_POST['op'] == t('Delete')) || isset($_POST['confirm'])) {
    return drupal_get_form('autolink_confirm_link_delete', $lid);
  }
  Autolink::loadFiles(array('links'));
  $link = autolink_get('LinkMapper')->load($lid);
  return drupal_get_form('autolink_form_link', $group, (array)$link);
}

/**
 * Form builder for the link delete confirmation form.
 *
 * @param $lid
 *   The ID of a link.
 *
 * @ingroup forms
 * @see autolink_confirm_link_delete_submit()
 */
function autolink_confirm_link_delete(&$form_state, $lid) {
  $data = autolink_get('LinkMapper')->load($lid);

  $form['type'] = array(
    '#type' => 'value',
    '#value' => 'link',
  );
  $form['lid'] = array(
    '#type' => 'value',
    '#value' => $data['lid'],
  );
  $form['keyword'] = array(
    '#type' => 'value',
    '#value' => $data['keyword'],
  );
  return confirm_form($form,
                  t('Are you sure you want to delete the link %keyword?',
                  array('%keyword' => $data['keyword'])),
                  "admin/content/autolink",
                  t('This action cannot be undone.'),
                  t('Delete'),
                  t('Cancel'));
}

/**
 * Submit handler to delete a link after confirmation.
 *
 * @see autolink_confirm_link_delete()
 */
function autolink_confirm_link_delete_submit($form, &$form_state) {
  // Delete the link and record the deletion
  $status = autolink_get('LinkMapper')->delete($form_state['values']['lid']);

  drupal_set_message(t('Deleted link definition for keyword <i>%link</i>.', array('%link' => $form_state['values']['keyword'])));
  watchdog('autolink', 'Deleted link definition for keyword <i>%link</i>.', array('%link' => $form_state['values']['keyword']), WATCHDOG_NOTICE);
  $form_state['redirect'] = 'admin/content/autolink';

  return;
}

/**
 * Merges AHAH form values into a readable $link array in preparation for database insert.
 *
 * @return
 *   Merged form values that can be processed for validation or inserted into the database.
 */
function _autolink_merge_link_form_values($form_values) {
  // Because of AHAH we need to add the proper form elements to a link array of form data.
  $link = $form_values;
  $link += $form_values['link_wrapper'];
  $link += $form_values['link_wrapper']['definition'];

  foreach ($form_values['link_wrapper']['definition'] as $key => $value) {
    if ($key == 'method_wrapper' || $value['#type'] == 'fieldset') {
      $link += $form_values['link_wrapper']['definition'][$key];
    }
  }

  return $link;
}
