<?php

/**
 * Autolink provides hooks for defining plugins and link types to the
 * Autolink API. These plugins and link types are built by extending
 * base classes. This means that they require a knowledge of object-
 * oriented PHP programming. Following are definitions and explanaitions
 * of how you can interact with and customize the Autolink module with
 * your own code.
 */

/**
 * Defines a plugin to the Autolink API.
 *
 * @return
 *   An associative array of plugin information. The key of the array
 *   should be the machine-name of the plugin, with only lower-case
 *   letters and underscores. The key of the array should be a machine-
 *   name of the plugin containing only lower-case letters and underscores.
 *   The machine-name should be prefixed by the implementing module's name
 *   to prevent name clashes in URLs and other plugins.
 *   - label: A translated label for use in administrative pages.
 *   - description: A translated string describing the primary linking
 *     functionality of the plugin. This string is used in administrative
 *     plugin lists in filter configuration overviews.
 *   - module: The name of the module that implements the plugin. This
 *     item will be automatically populated with the machine-name of the
 *     implementing module if not explicitly set.
 *   - processor: The machine-name of the processor that the plugin uses.
 *     This is required for Autolink to load the proper processor. Processors
 *     are defined in hook_autolink_processor_info().
 *   - default settings: An array of default settings with which to populate
 *     the plugin's settings before the plugin has been properly configured.
 *     See 'default settings callback' below for an alternative.
 *
 *   - The following values identify various administrative and processing callbacks.
 *     Implementing modules may store additional callbacks here and use the
 *     AutolinkPlugin::get_callback() method to retrieve the callback from plugin
 *     info. Also, modules can use the plugin's magic AutolinkPlugin::__call()
 *     method to call callbacks. Use the prefix execute_ and the callback name to
 *     automatically retrieve the actual callback function and get its return value.
 *     Using this method will search for a callback with the given name (after execute_)
 *     in the plugin info. So, if AutolinkPlugin::execute_validate() is called then
 *     the plugin will search for a key in plugin info named 'validate callback' and
 *     call that function if found. This is demostrated in the filter's process
 *     operation. Note: for each of these callbacks plugins may also use the form
 *     MODULE_PLUGIN_CALLBACK() without defining the callback here. For example, 
 *     autolink_keyword_autolink_node_keyword_form() can be used as a form callback
 *     for the autolink_node_keyword plugin defined by autolink_keyword.
 *     Here are the callbacks known to and called by Autolink:
 *   - preprocess callback: This value identifies a preprocess callback which is invoked
 *     before the actual processing is done. It passes the processor object to
 *     allow plugins to set settings on the processor before operations begin.
 *   - replace callback: This callback should always be implemented by plugins. Processors
 *     will call this callback to retrieve the proper replacement data when a
 *     match is found.
 *   - settings callback: The administrative plugin settings form callback. If no value
 *     is set Autolink will use module_plugin_form().
 *   - validate callback: A validation callback for the administrative plugin settings
 *     form. If no value is given Autolink will look for module_plugin_validate.
 *   - default settings callback: Allows plugins to specify default settings that
 *     are vital to the plugin's processing. This helps prevent errors in the event
 *     that content is filtered before any settings have been saved for the plugin.
 */
function hook_autolink_plugin_info() {
  return array(
    'autolink_keyword_node' => array(
      'label' => t('Keyword links plugin'),
      'description' => t('Converts pre-defined keywords to links in content.'),
      'processor' => 'keyword',
      'callbacks' => array(
        'preprocess' => 'autolink_keyword_preprocess',
        'replace' => 'autolink_keyword_replace',
        'form' => 'autolink_keyword_form_settings',
        'validate' => 'autolink_keyword_form_validate',
      ),
    ),
  );
}

/**
 * Alters plugin info. Plugins are passed by reference in an array.
 */
function hook_autolink_plugin_info_alter(&$plugins) {
  $plugins['autolink_keyword_node']['callbacks']['form'] = 'autolink_keyword_form';
}

/**
 * Preprocess callback.
 *
 * The preprocess callback allows plugins to change processor settings
 * before processing happens.
 *
 * @param $processor
 *   An AutolinkProcessor object to be modified. This object is passed by
 *   reference so any modifications made to its properties will apply in
 *   processing. The plugin object can be referenced at $processor->plugin.
 * @param $settings
 *   An associative array of plugin settings as set in the administrative
 *   interface by forms defined by the plugin. If no configuration form is
 *   defined by the plugin this will be an empty array.
 * @param $node
 *   The node to which the text being filtered belongs. Autolink filters will
 *   only process text if it can be related to a node.
 * @param $cache
 *   An Autolink plugin cache object. This makes interacting with the Autolink
 *   cache easier for plugins. Items may be stored in the Autolink cache by using
 *   the AutolinkCache::set_items() method.
 * @return
 *   None.
 */
function autolink_keyword_preprocess(AutolinkProcessor $processor, $settings, $node, AutolinkCache $cache) {
  // Set item limits.
  $processor->total_limit = AUTOLINK_DEFAULT_LIMIT;
  $processor->item_limit = $settings['item_limit'];

  // Tell the processor how to validate information.
  $processor->validate_items = FALSE;
  $processor->validate_limits = TRUE;

  // Add items to the processor.
  if ($items = $cache->get_items($node->type)) {
    $processor->add_items($items->data);
  }
  else {
    $links = autolink_keyword_items($node);
    foreach ($links as $item) {
      $processor->add_item($item);
    }
  }

  // Cache items for use next time.
  // Items can be stored in the cache with a unique identifier link node type as the first parameter.
  $cache->set_items($node->type, $links);
}

/**
 * Replace callback.
 *
 * @param $match
 *   The text that was matched by the processor. This can vary between processors.
 *   With regard to existing processors, the keyword processor will return a matched
 *   keyword as defined in items and the pattern processor will pass a matched pattern
 *   as defined in the class' regular expression property.
 * @param $settings
 *   An array of plugin settings defined in plugin administration pages.
 * @param $item
 *   For some processors, items that are set in the processor are processed individually.
 *   For instance, the keyword processor will pass an item to retrieve a replacement from
 *   the current plugin. If the processor does not use items then an empty array is passed.
 * @return
 *   A replacement link or other value to be placed in filtered text.
 */
function autolink_keyword_replace($match, $settings, $item = array()) {
  return $item['replace'];
}

/**
 * Configuration form callback.
 *
 * Upon submission of this form, settings are automatically saved by Autolink
 * with field machine-names as keys. All checkboxes fields are filtered before
 * saves so only enabled values will be saved.
 *
 * @param $form
 *   The plugin settings form array. This argument is passed by reference
 *   so plugin modules may alter it directly.
 * @param $settings
 *   An associative array of settings that have been saved in the plugin
 *   configuration form. Plugins should use conditionals to check that values
 *   are set when setting default values. If no settings have been set this
 *   argument will be an empty array.
 * @param $plugin
 *   The machine-name of the plugin whose form is being built. This argument
 *   is passed to allow reuse of the same form builder callback for different
 *   plugins.
 * @param $format
 *   The input format currently being altered.
 * @return
 *   None.
 */
function autolink_keyword_form_settings(&$form, $settings, $plugin, $format) {
  $form['keywords'] = array(
    '#type' => 'textfield',
    '#title' => t('Keywords'),
  );
}

/**
 * Defines a processor class.
 *
 * Data processors may be defined to Autolink to handle specific data
 * in a unique way. View existing Autolink processors for more info.
 *
 * @return
 *   An array of information about a data processor. The key of the
 *   array is the machine-name of the processor.
 *   - label: A human-readable label for the processor.
 *   - module: The defining module of the processor. This is used to
 *     determine the base path from which to load the file.
 *   - class: The processor class.
 */
function hook_autolink_processor_info() {
  return array(
    'default' => array(
      'label' => t('Default processor'),
      'module' => 'autolink',
      'class' => 'AutolinkNodeProcessor',
    ),
  );
}

/**
 * Alters existing processor data.
 */
function hook_autolink_processor_info_alter(&$processors) {
  $processors['node']['class'] = 'AutolinkNodeProcessor';
}

/**
 * Defines a processor class.
 *
 * This sample is taken from the AutolinkTextProcessor class.
 * Note that AutolinkProcessor does use a __call() method, which calls
 * plugin methods, so calls to plugin methods can be accomplished by
 * making direct method calls like $this->getFields().
 *
 * Note that the AutolinkProcessor class is abstract and requires that
 * child classes implement the following methods:
 * public function save();
 * protected function process($links);
 * public function clear($limit = AUTOLINK_DEFAULT_LIMIT, $link = NULL);
 */
class AutolinkSampleProcessor extends AutolinkProcessor {
  var $expression = '/(?<!\\\\)\[(.+)]/ie';

  /**
   * Processes text that is currently being filtered.
   *
   * This method is an extension of an abstract method, so its signature
   * must match. Processors may do whatever they need to to generate items
   * in text. The parent AutolinkProcessor class offers many features to
   * properly integrate with plugins. If possible, processors should keep
   * track of link generation counts by using the AutolinkProcessor::add_count()
   * method as well as AutolinkProcessor::valid_limit() to check to ensure
   * that limits imposed by the current plugin are not surpassed. Processors
   * should *always* use validation methods as they are configurable by plugins.
   * Plugins can ignore validation methods by settings processor properties.
   *
   * @param $text
   *   The text being filtered. This parameter is passed by reference to be
   *   altered directly.
   * @param $node
   *   The node to which the text being processed belongs.
   * @param $items
   *   An array of items that have been added by plugins. If no items have
   *   been added to the items array then this will be an empty array.
   */
  public function execute(&$text, $node, array $items) {
    foreach ($items as $id => $item) {
      if (!$this->valid_limit()) {
        break;
      }
      elseif ($this->valid_item($item) && $this->valid_nid($node, $item)) {
        $text = preg_replace(
          $this->expression,
          // The get_replacement method automatically calls the plugin's replace callback.
          '$this->get_replacement("$1", $item)',
          $text,
          $this->get_item_limit(),
          $count
        );
        // Track the count of replacements made for plugins that impose limits.
        if (!empty($count)) {
          $this->add_count($count);
        }
      }
    }
  }

}

/**
 * Identifies processing files for Autolink to load during processing.
 * Files are lazy loaded only during filtering events.
 *
 * @return
 *   An array of values identifying files to load. These files should
 *   be from the base path of the implementing module.
 */
function hook_autolink_file_info() {
  return array('plugin.inc', 'processor.inc');
}
