<?php

/**
 * @file
 * Autolink text processors.
 */

/**
 * Define the cache interface.
 */
interface AutolinkPluginCache {
  public function set_items($id, $items);
  public function get_items($id);
  public function clear_items($id);
}

/**
 * Cache object class handles caching within processors. During preprocessing,
 * the cache object is passed to plugin callbacks, allowing them to easily
 * cache data in the Autolink cache.
 */
class AutolinkCache implements AutolinkPluginCache {

  // The machine-name of the current plugin. This is used in building
  // cache IDs for unique storage.
  var $plugin;

  // The current input format. This is also used for building cache IDs.
  var $format;

  /**
   * Standard constructor method.
   */
  public function __construct($plugin, $format) {
    $this->plugin = $plugin;
    $this->format = $format;
  }

  /**
   * Caches item arrays for specific plugins. This should be used by plugins
   * within the scope of the preprocess callback.
   */
  public function set_items($id, $items, $expire = CACHE_PERMANENT) {
    if (!empty($items) && is_array($items)) {
      cache_set($this->plugin . '_' . $this->format . '_items_' . $id, $items, AUTOLINK_CACHE, $expire);
    }
  }

  /**
   * Returns cached items from the Autolink cache. This may be used by plugins
   * within the scope of the preprocess callback.
   */
  public function get_items($id) {
    return cache_get($this->plugin . '_' . $this->format . '_items_' . $id, AUTOLINK_CACHE);
  }

  /**
   * Clears the items cache for a specific plugin.
   */
  public function clear_items($id) {
    cache_clear_all($this->plugin . '_' . $this->format . '_items_' . $id, AUTOLINK_CACHE);
  }

}

/**
 * Defines the Autolink plugin interface.
 */
interface AutolinkPluginObject {
  public function init($plugin, $format);
  public function get_info();
  public function get_callback($name);
  public function save();
  public function reset();
}

/**
 * Plugins handle actual processing of text and generation of links.
 */
class AutolinkPlugin implements AutolinkPluginObject {

  // An AutolinkConfig object populated with settings for the current input format.
  protected $config;

  // The AutolinkCache object related to this plugin. Will handle caching data for the plugin.
  public $cache;

  // A string indicating the current input format.
  public $format;

  // An associative array of plugin information defined in hook_autolink_plugin_info().
  public $info = array();

  // The machine-name of the plugin.
  public $name = NULL;

  // The module that defines the plugin.
  public $module = NULL;

  // The human-readable name of the plugin.
  public $label = '';

  // The human-readable description of the plugin.
  public $description = '';

  // A string identifying the processor that is used by this plugin.
  public $processor = NULL;

  // An array of configuration settings as saved in plugin configuration pages
  // via the Drupal input formats interface.
  public $settings = array();

  public function __construct() {}

  /**
   * Initializes a plugin instance. Properties are populated from plugin info.
   */
  public function init($plugin, $format) {
    $config = autolink_config($format);
    $this->format = $format;
    if ($plugin_info = $config::$plugin_info[$plugin]) {
      $this->config = $config;
      $this->info = $plugin_info;
      $this->name = $plugin;
      $this->module = $plugin_info['module'];
      $this->cache = new AutolinkCache($plugin, $format);
      foreach ($plugin_info as $property => $value) {
        $this->{$property} = $value;
      }

      // If no settings have been saved yet then check for default settings.
      // It is possible for a plugin to define default settings from both within
      // plugin information and in a separate callback. The callback is called last.
      $this->settings = variable_get('autolink_plugin_' . $this->name . '_' . $this->format, array());
      if (empty($this->settings)) {
        if (isset($this->info['default settings'])) {
          $this->settings = $this->info['default settings'];
        }
        if (isset($this->info['default settings callback']) && function_exists($this->info['default settings callback'])) {
          $defaults = call_user_func($this->info['default settings callback'], $plugin, $format);
          $this->settings = array_merge($this->settings, $defaults);
        }
      }
    }
    return $this;
  }

  /**
   * Magic setter method.
   */
  public function __set($property, $value) {
    $this->{$property} = $value;
  }

  /**
   * Magic call method for calling functions based on plugin info.
   *
   * This allows plugin modules to define their own callbacks and call
   * it by using $plugin->execute_(callback). We search for 'execute_'
   * in the method and if found strip it and get the proper callback
   * from the plugin info. If no callback was identified then
   * {module}_{plugin}_{callback_name} is used.
   *
   * Since this magic callback is used many times during processing,
   * we statically cache functions that do not exist to prevent having
   * to repeat the check dozens of times in the same process.
   */
  public function __call($method, $args) {
    static $callbacks = array();
    // If the callback exists and is known it will be stored in the static array.
    if (isset($callbacks[$this->name][$method])) {
      // Callbacks stored in the static variable have already been checked.
      if (!empty($callbacks[$this->name][$method])) {
        return call_user_func_array($callbacks[$this->name][$method], $args);
      }
      // The callback was not found previously so return nothing.
      return;
    }
    // We haven't checked for this callback yet, so parse the method and check for it.
    elseif (strstr($method, 'execute_')) {
      $name = substr($method, 8);
      $callback = $this->get_callback($name);
      if (function_exists($callback)) {
        $callbacks[$this->name][$method] = $callback;
        return call_user_func_array($callback, $args);
      }
      // Callbacks that don't exist are noted in the static array as FALSE.
      else {
        $callbacks[$this->name][$method] = FALSE;
      }
    }
  }

  /**
   * Returns a callback string as identified in plugin information.
   *
   * Plugin modules may use this method to retrieve accurate callback information
   * defined in custom values in hook_autolink_plugin_info(). Any plugin can simply
   * add another key with the word 'callback' following the callback name and use
   * this method to retrieve that information. Also see the magic AutolinkPlugin::__call()
   * method for more use of this method.
   *
   * @param $name
   *   The prefix of the callback to retrieve as identified in plugin information.
   * @return
   *   The callback located in plugin information if it exists, otherwise, the method
   *   will return an alternative callback based on standard Autolink convention.
   *   This callback is formatted {MODULE}_{PLUGIN}_{CALLBACK}.
   * @see AutolinkPlugin::__call()
   */
  public function get_callback($name) {
    return isset($this->info[$name . ' callback']) ? $this->info[$name . ' callback'] : "{$this->module}_{$this->name}_{$name}";
  }

  /**
   * Returns plugin info.
   */
  public function get_info() {
    return self::$info[$this->name];
  }

  /**
   * Saves plugin settings.
   */
  public function save() {
    $this->settings['weight'] = $this->weight;
    variable_set('autolink_plugin_' . $this->name . '_' . $this->format, $this->settings);
  }

  /**
   * Resets the plugin settings.
   */
  public function reset() {
    $this->settings = array();
  }

}
