<?php

/**
 * @file
 * Administrative forms for autolink_ module.
 */

/**
 * Settings form for autolink.
 *
 * @ingroup forms
 * @see autolink_form_settings_submit()
 */
function autolink_form_settings($form, &$form_state, $edit = array()) {
  return autolink_get('config')->setForm()->execute('form', $form_state, $edit);
}

/**
 * Validation function for settings form.
 */
function autolink_form_settings_validate($form, &$form_state) {
  return autolink_get('config')->setForm()->execute('validate', $form, $form_state);
}

/**
 * Submit handler for Autolink settings form.
 *
 * This handler sets miltiple value field values by creating individual variables
 * for each value. This makes node and link type settings much easier to work with
 * on other forms and in processing functions and is cleaner than using the
 * 'array_filter' field.
 */
function autolink_form_settings_submit($form, &$form_state) {
  autolink_get('config')->setForm()->execute('submit', $form, $form_state);
}

/**
 * Callback for displaying the groups overview form.
 */
function autolink_admin_groups() {
  Autolink::loadFiles(array('filters', 'operations'));
  if (!empty($_POST['groups']) && isset($_POST['operation']) && ($_POST['operation'] == 'delete')) {
    $build['autolink_group_multiple_delete_confirm'] = drupal_get_form('autolink_group_multiple_delete_confirm');
  }
  else {
    $build['autolink_filter_form'] = drupal_get_form('autolink_filter_form');
    $build['autolink_overview_groups'] = drupal_get_form('autolink_overview_groups');
  }
  return $build;
}

/**
 * Form builder for autolink_ groups overview filters.
 *
 * @ingroup forms
 * @see autolink_filter_form_submit()
 */
function autolink_filter_form() {
  $session = isset($_SESSION['autolink_overview_filter']) ? $_SESSION['autolink_overview_filter'] : array();
  $filters = autolink_filters();

  $i = 0;
  $form['filters'] = array(
    '#type' => 'fieldset',
    '#title' => t('Show only link groups where'),
    '#theme' => 'exposed_filters__group',
  );
  foreach ($session as $filter) {
    list($type, $value) = $filter;
    $value = $filters[$type]['options'][$value];
    $t_args = array('%property' => $filters[$type]['title'], '%value' => $value);
    if ($i++) {
      $form['filters']['current'][] = array('#markup' => t('and where %property is %value', $t_args));
    }
    else {
      $form['filters']['current'][] = array('#markup' => t('%property is %value', $t_args));
    }
  }

  $form['filters']['status'] = array(
    '#type' => 'container',
    '#attributes' => array('class' => array('clearfix')),
    '#prefix' => ($i ? '<div class="additional-filters">' . t('and where') . '</div>' : ''),
  );
  $form['filters']['status']['filters'] = array(
    '#type' => 'container',
    '#attributes' => array('class' => array('filters')),
  );
  foreach ($filters as $key => $filter) {
    $form['filters']['status']['filters'][$key] = array(
      '#type' => 'select',
      '#options' => $filter['options'],
      '#title' => $filter['title'],
      '#default_value' => '[any]',
    );
  }

  $form['filters']['status']['actions'] = array(
    '#type' => 'actions',
    '#attributes' => array('class' => array('container-inline')),
  );
  $form['filters']['status']['actions']['submit'] = array(
    '#type' => 'submit',
    '#value' => (count($session) ? t('Refine') : t('Filter')),
  );
  if (count($session)) {
    $form['filters']['status']['actions']['undo'] = array(
      '#type' => 'submit',
      '#value' => t('Undo'),
    );
    $form['filters']['status']['actions']['reset'] = array(
      '#type' => 'submit',
      '#value' => t('Reset'),
    );
  }

  drupal_add_library('system', 'drupal.form');

  return $form;
}

/**
 * Process result from autolink administration filter form.
 */
function autolink_filter_form_submit($form, &$form_state) {
  $op = $form_state['values']['op'];
  $filters = autolink_filters();

  switch ($op) {
    case t('Filter'):
    case t('Refine'):
      foreach ($filters as $filter => $options) {
        if (isset($form_state['values']['filter'])) {
          $filter = $form_state['values']['filter'];
          $options = $filters[$filter]['options'];
          if (isset($options[$form_state['values'][$filter]])) {
            $_SESSION['autolink_overview_filter'][] = array($filter, $form_state['values'][$filter]);
          }
        }
      }
      break;
    case t('Undo'):
      array_pop($_SESSION['autolink_overview_filter']);
      break;
    case t('Reset'):
      $_SESSION['autolink_overview_filter'] = array();
      break;
    case t('Update'):
      return;
  }

  $form_state['redirect'] = 'admin/config/content/autolink/';
  return;
}

/**
 * Form builder; autolink_ administration page.
 *
 * @ingroup forms
 * @see autolink_overview_groups_validate()
 * @see autolink_overview_groups_submit()
 */
function autolink_overview_groups() {
  $header = array(
    'name' => array('data' => t('Group'), 'field' => 'g.name'),
    'description' => array('data' => t('Description'), 'field' => 'g.description'),
    'node_types' => array('data' => t('Node types')),
    'status' => array('data' => t('Status'), 'field' => 'g.status', 'sort' => 'desc'),
    'operations' => array('data' => t('Operations')),
  );

  $query = db_select('autolink_group', 'g');
  $query->leftJoin('autolink_group_node_type', 't', 'g.type = t.group_type');
  autolink_build_filter_query($query);

  $count_query = clone $query;
  $count_query->addExpression('COUNT(g.type)');

  $query = $query->extend('PagerDefault')->extend('TableSort');
  $query
    ->fields('g', array('type', 'name', 'description', 'status'), 't', array('node_type'))
    ->limit(25)
    ->orderByHeader($header)
    ->setCountQuery($count_query);
  $result = $query->execute();

  $form['options'] = array(
    '#type' => 'fieldset',
    '#title' => t('Update options'),
    '#attributes' => array('class' => array('container-inline')),
  );

  // Add our own operations first.
  $options = array();
  $ops = autolink_group_operations();
  foreach ($ops as $operation => $array) {
    $options[$operation] = $array['label'];
  }

  // Invoke hook_autolink_group_operations() to get a list of external operations.
  foreach (module_invoke_all('autolink_group_operations') as $operation => $array) {
    $options[$operation] = $array['label'];
  }

  $form['options']['operation'] = array(
    '#type' => 'select',
    '#title' => t('Operation'),
    '#title_display' => 'invisible',
    '#options' => $options,
    '#default_value' => 'activate',
  );
  $options = array();
  $form['options']['submit'] = array(
    '#type' => 'submit',
    '#value' => t('Update'),
  );

  $destination = drupal_get_destination();

  $status = array(t('disabled'), t('active'));
  $groups = array();
  foreach ($result as $group) {
    $node_types = array();
    $types_result = db_query('SELECT n.type, n.name FROM {node_type} n INNER JOIN {autolink_group_node_type} t ON n.type = t.node_type WHERE t.group_type = :type', array(':type' => $group->type));
    foreach ($types_result as $node_type) {
      $node_types[$node_type->type] = $node_type->name;
    }
    asort($node_types);

    $options[$group->type] = array(
      'name' => array('data' => array('#type' => 'link', '#title' => $group->name, '#href' => "admin/config/content/autolink/$group->type")),
      'description' => check_plain($group->description),
      'node_types' => theme('item_list', array('items' => $node_types)),
      'status' => $status[$group->status],
      'operations' => array('data' => array('#type' => 'link', '#title' => t('edit group'), '#href' => "admin/config/content/autolink/edit/group/$group->type", '#options' => array('query' => $destination))),
    );
  }

  $form['groups'] = array(
    '#type' => 'tableselect',
    '#header' => $header,
    '#options' => $options,
    '#empty' => t('No groups available.'),
  );
  $form['pager'] = array('#markup' => theme('pager'));

  return $form;
}

/**
 * Form validation for autolink_ groups administration update form.
 */
function autolink_overview_groups_validate($form, &$form_state) {
  $form_state['values']['groups'] = array_filter($form_state['values']['groups']);
  if (!count($form_state['values']['groups'])) {
    form_set_error('', t('No groups were selected.'));
  }
}

/**
 * Submit handler for autolink_ groups administration update form.
 */
function autolink_overview_groups_submit($form, &$form_state) {
  $operations = autolink_group_operations($form_state);
  $operations += module_invoke_all('autolink_group_operations', $form_state);
  $operation = $operations[$form_state['values']['operation']];
  // Filter out unchecked links.
  $groups = array_filter($form_state['values']['groups']);
  if ($function = $operation['callback']) {
    // Add in callback arguments if present.
    if (isset($operation['callback arguments'])) {
      $args = array_merge(array($groups), $operation['callback arguments']);
    }
    else {
      $args = array($groups);
    }
    // Call the function based on the operation that was selected.
    call_user_func_array($function, $args);

    drupal_set_message(t('The update has been performed.'));
  }
}

/**
 * Display form for adding and editing link groups.
 *
 * @ingroup forms
 * @see autolink_form_group_submit()
 */
function autolink_form_group($form, &$form_state, $edit = array()) {
  if (!isset($form_state['group'])) {
    $group = is_object($edit) ? $edit : (object) $edit;
    $defaults = array(
      'status' => 1,
      'name' => '',
      'type' => '',
      'description' => '',
      'types' => array(),
    );
    foreach ($defaults as $key => $value) {
      if (!isset($group->$key)) {
        $group->$key = $value;
      }
    }
    $form_state['group'] = $group;
  }
  else {
    $group = $form_state['group'];
  }

  $form['#group'] = $form_state['group'];

  // Check whether we need a deletion confirmation form.
  if (isset($form_state['confirm_delete']) && isset($form_state['values']['type'])) {
    return autolink_group_confirm_delete($form, $form_state, $form_state['values']['type']);
  }

  $form['identification'] = array(
    '#type' => 'fieldset',
    '#title' => t('Identification'),
    '#collapsible' => TRUE,
  );
  $form['identification']['status'] = array(
    '#type' => 'select',
    '#title' => t('Status'),
    '#default_value' => $group->status,
    '#options' => array(0 => t('Disabled'), 1 => t('Active')),
  );
  $form['identification']['name'] = array(
    '#type' => 'textfield',
    '#title' => t('Group name'),
    '#default_value' => $group->name,
    '#maxlength' => 255,
    '#description' => t('The human-readable name for this link group.'),
    '#required' => TRUE,
  );
  $form['identification']['description'] = array(
    '#type' => 'textarea',
    '#title' => t('Description'),
    '#default_value' => $group->description,
    '#description' => t('Description of the link group.'),
  );
  $form['identification']['nodes'] = array(
    '#type' => 'fieldset',
    '#title' => t('Node types'),
    '#collapsible' => TRUE,
  );
  $form['identification']['nodes']['types'] = array(
    '#type' => 'checkboxes',
    '#title' => t('Node types'),
    '#options' => node_type_get_names(),
    '#default_value' => $group->types,
  );

  $form['actions'] = array('#type' => 'actions');
  $form['actions']['submit'] = array('#type' => 'submit', '#value' => t('Save'));
  if (isset($group->type)) {
    $form['actions']['delete'] = array('#type' => 'submit', '#value' => t('Delete'));
    $form['type'] = array('#type' => 'value', '#value' => $group->type);
  }

  if (!isset($group->type)) {
    $form['#validate'][] = 'autolink_form_group_duplicate_validate';
  }

  return $form;
}

/**
 * Validate function for add link group form.
 */
function autolink_form_group_duplicate_validate($form, &$form_state) {
  $group = $form_state['values']['name'];
  $result = db_query("SELECT * FROM {autolink_group} WHERE name = :group", array(':group' => $group))->fetchObject();
  if ($result) {
    form_set_error('name', t('A link group with the name %name already exists.', array('%name' => $group)));
  }
}

/**
 * Submit function for add link group form.
 */
function autolink_form_group_submit($form, &$form_state) {
  if ($form_state['clicked_button']['#value'] == t('Delete')) {
    // Rebuild the form to confirm group deletion.
    $form_state['rebuild'] = TRUE;
    $form_state['confirm_delete'] = TRUE;
    return;
  }

  $group = $form_state['values'];
  switch (autolink_group_save($group)) {
    case SAVED_NEW:
      drupal_set_message(t('Created new link group %name.', array('%name' => $group->name)));
      watchdog('autolink', 'Created new link group %name.', array('%name' => $group->name), WATCHDOG_NOTICE, l(t('edit'), 'admin/config/content/autolink/' . $group->type . '/edit'));
      break;

    case SAVED_UPDATED:
      drupal_set_message(t('Updated link group %name.', array('%name' => $group->name)));
      watchdog('autolink', 'Updated link group %name.', array('%name' => $group->name), WATCHDOG_NOTICE, l(t('edit'), 'admin/config/content/autolink/' . $group->type . '/edit'));
      break;
  }

  $form_state['values']['type'] = $group->type;
  $form_state['type'] = $group->type;
  $form_state['redirect'] = 'admin/config/content/autolink';
}

/**
 * Page to edit a link group.
 */
function autolink_admin_group_edit($group) {
  if ((isset($_POST['op']) && $_POST['op'] == t('Delete')) || isset($_POST['confirm'])) {
    return drupal_get_form('autolink_group_confirm_delete', $group->type);
  }
  return drupal_get_form('autolink_form_group', (array)$group);
}

/**
 * Form builder for the link group delete confirmation form.
 *
 * @ingroup forms
 * @see autolink_group_confirm_delete_submit()
 */
function autolink_group_confirm_delete($form, &$form_state, $type) {
  $group = autolink_group_load($type);

  $form['type'] = array('#type' => 'value', '#value' => $type);

  $form['#group'] = $group;
  $form['#id'] = 'autolink_group_confirm_delete';
  $form['type'] = array('#type' => 'value', '#value' => 'group');
  $form['name'] = array('#type' => 'value', '#value' => $group->name);
  $form['#submit'] = array('autolink_group_confirm_delete_submit');
  return confirm_form($form,
    t('Are you sure you want to delete the link group %title?',
    array('%title' => $group->name)),
    'admin/config/content/autolink',
    t('Deleting a link group will delete all the links in it. This action cannot be undone.'),
    t('Delete'),
    t('Cancel'));
}

/**
 * Submit handler to delete a link group after confirmation.
 *
 * @see autolink_group_confirm_delete()
 */
function autolink_group_confirm_delete_submit($form, &$form_state) {
  $status = autolink_group_delete($form_state['values']['type']);
  drupal_set_message(t('Deleted link group %name.', array('%name' => $form_state['values']['name'])));
  watchdog('autolink', 'Deleted link group %name.', array('%name' => $form_state['values']['name']), WATCHDOG_NOTICE);
  $form_state['redirect'] = 'admin/config/content/autolink';
  cache_clear_all();
  return;
}

/**
 * Callback for generating the links overview form.
 */
function autolink_admin_links($callback_arg = '', $group) {
  Autolink::loadFiles(array('filters', 'operations'));
  if (!empty($_POST['links']) && isset($_POST['operation']) && ($_POST['operation'] == 'delete')) {
    $build['autolink_link_multiple_delete_confirm'] = drupal_get_form('autolink_link_multiple_delete_confirm');
  }
  else {
    $build['autolink_overview_links'] = drupal_get_form('autolink_overview_links', $group->type);
  }
  return $build;
}

/**
 * Form builder; autolink_ administration page.
 *
 * @ingroup forms
 * @see autolink_overview_links_validate()
 * @see autolink_overview_links_submit()
 */
function autolink_overview_links($form, &$form_state, $group) {
  $header = array(
    'type' => array('data' => t('Type'), 'field' => 'a.type'),
    'keyword' => array('data' => t('Keyword'), 'field' => 'a.keyword'),
    'destination' => array('data' => t('Destination')),
    'operations' => array('data' => t('Operations')),
  );

  $query = db_select('autolink_link', 'a');
  $query->condition('a.group_type', $group, '=');

  $count_query = clone $query;
  $count_query->addExpression('COUNT(a.lid)');

  $query = $query->extend('PagerDefault')->extend('TableSort');
  $query
    ->fields('a', array('lid', 'group_type', 'type', 'keyword', 'path', 'query'))
    ->limit(25)
    ->orderByHeader($header)
    ->setCountQuery($count_query);
  $result = $query->execute();
  $form['options'] = array(
    '#type' => 'fieldset',
    '#title' => t('Update options'),
    '#attributes' => array('class' => array('container-inline')),
  );
  $form['options'] = array(
    '#type' => 'fieldset',
    '#title' => t('Update options'),
    '#attributes' => array('class' => array('container-inline')),
  );

  // Add our own operations first.
  $ops = autolink_link_operations();
  foreach ($ops as $operation => $array) {
    $options[$operation] = $array['label'];
  }

  // Invoke hook_autolink_operations() to get a list of external operations.
  foreach (module_invoke_all('autolink_link_operations') as $operation => $array) {
    $options[$operation] = $array['label'];
  }

  $form['options']['operation'] = array(
    '#type' => 'select',
    '#title' => t('Operation'),
    '#title_display' => 'invisible',
    '#options' => $options,
    '#default_value' => 'unblock',
  );
  $form['options']['submit'] = array(
    '#type' => 'submit',
    '#value' => t('Update'),
  );

  $options = array();
  $link_types = autolink_get('LinkTypeInfo')->getDefinable('label');
  $destination = drupal_get_destination();
  foreach ($result as $link) {
    $path = drupal_lookup_path('alias', $link->path) ? drupal_lookup_path('alias', $link->path) : $link->path;
    $options[$link->lid] = array(
      'type' => check_plain($link->type),
      'keyword' => $link->keyword,
      'destination' => check_plain($path),
      'operations' => array('data' => array('#type' => 'link', '#title' => t('edit link'), '#href' => "admin/config/content/autolink/$link->group_type/edit/link/$link->lid", '#options' => array('query' => $destination))),
    );
  }

  $form['links'] = array(
    '#type' => 'tableselect',
    '#header' => $header,
    '#options' => $options,
    '#empty' => t('No links available.'),
  );
  $form['pager'] = array('#markup' => theme('pager'));
  $form['group'] = array('#type' => 'value', '#value' => $group);

  return $form;
}

/**
 * Form validation for autolink_ administration update form.
 */
function autolink_overview_links_validate($form, &$form_state) {
  $form_state['values']['links'] = array_filter($form_state['values']['links']);
  if (!count($form_state['values']['links'])) {
    form_set_error('', t('No link definitions were selected.'));
  }
}

/**
 * Submit handler for autolink_ administration update form.
 */
function autolink_overview_links_submit($form, &$form_state) {
  $operations = autolink_link_operations($form_state);
  $operations += module_invoke_all('autolink_link_operations', $form_state);
  $operation = $operations[$form_state['values']['operation']];
  // Filter out unchecked links.
  $links = array_filter($form_state['values']['links']);
  if ($function = $operation['callback']) {
    // Add in callback arguments if present.
    if (isset($operation['callback arguments'])) {
      $args = array_merge(array($links), $operation['callback arguments']);
    }
    else {
      $args = array($links);
    }
    // Call the function based on the operation that was selected.
    call_user_func_array($function, $args);

    drupal_set_message(t('The update has been performed.'));
  }
}

/**
 * Form for adding or editing a link definition.
 *
 * This function uses the AHAH Helper module to build a dynamic link definition
 * form. When the AHAH submit function is called and the form is rebuilt this
 * function builds link type specific form elements based on data retrieved with
 * the link type that has been selected.
 */
function autolink_form_link($form, &$form_state, $group, $settings = array()) {
  // Load JS and CSS files for hiding the 'Select' button for AJAX enabled forms.
  drupal_add_js(drupal_get_path('module', 'autolink') . '/autolink.js', 'file');
  drupal_add_css(drupal_get_path('module', 'autolink') . '/autolink.css');

  $form = array();
  $config = autolink_get('Config');
  $form['group'] = array('#type' => 'value', '#value' => $group->type);

  if (!isset($form_state['values']['method'])) {
    $method = isset($settings['method']) ? $settings['method'] : 'select';
  }
  else {
    $method = $form_state['values']['method'];
  }

  // Prepare link types for the form.
  $method_options = array('select' => t('--Select a link type--'));
  $methods = $config->getFormMethods();

  if (!empty($methods)) {
    $method_options += $methods;
  }
  else {
    drupal_set_message(t('There are currently no definable link types enabled. Link types may be enabled by visiting the plugin settings page.'), 'warning');
  }

  // If there is only one link type then set that as the default instead of 'select'.
  $form['method'] = array(
    '#type' => 'select',
    '#title' => t('Link type'),
    '#options' => $method_options,
    '#default_value' => $method,
    '#ajax' => array(
      'event' => 'change',
      'wrapper' => 'autolink-wrapper',
      'callback' => 'autolink_build_link_form',
      'method' => 'replace',
    ),
  );

  $form['method_submit'] = array(
    '#type' => 'submit',
    '#value' => t('Select'),
    '#attributes' => array('class' => array('select-button')),
  );

  $form['link_wrapper'] = array(
    '#prefix' => '<div id="autolink-wrapper">',
    '#suffix' => '</div>',
  );

  // Only if a link type has been selected do we show the link definition form.
  if (!empty($method) && $method != 'select') {
    $parts = explode('-', $method);
    $link_type = $parts[0];
    $form['type'] = array('#type' => 'value', '#value' => $link_type);

    $form['link_wrapper']['definition'] = array(
      '#type' => 'fieldset',
      '#title' => t('Definition'),
      '#collapsible' => TRUE,
    );

    $type_form = AutolinkFormFactory::getInstance($method);
    $form['link_wrapper']['definition'] += $type_form->execute('form', $settings);
  }

  $form['submit'] = array('#type' => 'submit', '#value' => t('Save'));
  $form['group_type'] = array('#type' => 'value', '#value' => $group->type);
  if (isset($settings['lid'])) {
    $form['delete'] = array('#type' => 'submit', '#value' => t('Delete'));
    $form['lid'] = array('#type' => 'value', '#value' => $settings['lid']);
  }
  return $form;
}

/**
 * AJAX callback for the link add/edit form.
 */
function autolink_build_link_form($form, $form_state) {
  return $form['link_wrapper'];
}

/**
 * Validation handler for the link edit form.
 *
 * Here we ensure the link does not contain inappropriate characters. Also, since
 * we are using disabled and hidden fields, we have to do extra form validation
 * here because we cannot use the 'required' => TRUE property on a disabled field.
 *
 * @see autolink_form_link()
 */
function autolink_form_link_validate($form, &$form_state) {
  $values = _autolink_merge_link_form_values($form_state['values']);
  AutolinkFormFactory::getInstance($values['method'])->execute('validate', $values);
}

/**
 * Final submit function for the link form.
 */
function autolink_form_link_submit($form, &$form_state) {
  $values = _autolink_merge_link_form_values($form_state['values']);
  $object = AutolinkFormFactory::getInstance($values['method']);

  switch ($object->execute('save', $values)) {
    case SAVED_NEW:
      drupal_set_message(t('Created new <i>%link_type</i> type link for keyword <i>%keyword</i>.', array('%link_type' => $values['type'], '%keyword' => $values['keyword'])));
      watchdog('autolink', 'Created new link for keyword <i>%keyword</i>.', array('%keyword' => $values['keyword']), WATCHDOG_NOTICE, l(t('edit'), 'admin/content/autolink/' . $values['group_type'] . '/edit/link/' . $values['lid']));
      break;
    case SAVED_UPDATED:
      drupal_set_message(t('Updated <i>%link_type</i> type link for keyword <i>%keyword</i>.', array('%link_type' => $form_state['values']['link_wrapper']['type'], '%keyword' => $values['stored_keyword'])));
      watchdog('autolink', 'Updated link for keyword <i>%keyword</i>.', array('%keyword' => $values['stored_keyword']), WATCHDOG_NOTICE, l(t('edit'), 'admin/content/autolink/'. $values['group_type'] . '/edit/link/' . $link['lid']));
      break;
  }

  $form_state['lid'] = isset($form_state['values']['lid']) ? $form_state['values']['lid'] : '';
  drupal_goto('admin/config/content/autolink/' . $values['group_type']);

  return;
}

/**
 * Page to add a link.
 */
function autolink_add_link_page($group) {
  drupal_set_title(t("Add link definition to $group->name"));
  return drupal_get_form('autolink_form_link', $group);
}

/**
 * Page to edit a link.
 *
 * @param $link
 *   A link object.
 *
 * @see autolink_form_link()
 * @see autolink_confirm_delete()
 */
function autolink_admin_link_edit($group, $lid) {
  if ((isset($_POST['op']) && $_POST['op'] == t('Delete')) || isset($_POST['confirm'])) {
    return drupal_get_form('autolink_confirm_link_delete', $lid);
  }
  $link = autolink_get('LinkMapper')->load($lid);
  return drupal_get_form('autolink_form_link', $group, (array)$link);
}

/**
 * Form builder for the link delete confirmation form.
 *
 * @param $lid
 *   The ID of a link.
 *
 * @ingroup forms
 * @see autolink_confirm_link_delete_submit()
 */
function autolink_confirm_link_delete($form, &$form_state, $lid) {
  $data = autolink_get('LinkMapper')->load($lid);

  $form['type'] = array(
    '#type' => 'value',
    '#value' => 'link',
  );
  $form['lid'] = array(
    '#type' => 'value',
    '#value' => $data['lid'],
  );
  $form['keyword'] = array(
    '#type' => 'value',
    '#value' => $data['keyword'],
  );
  return confirm_form($form,
                  t('Are you sure you want to delete the link %keyword?',
                  array('%keyword' => $data['keyword'])),
                  "admin/config/content/autolink/$link->group_type",
                  t('This action cannot be undone.'),
                  t('Delete'),
                  t('Cancel'));
}

/**
 * Submit handler to delete a link after confirmation.
 *
 * @see autolink_confirm_link_delete()
 */
function autolink_confirm_link_delete_submit($form, &$form_state) {
  // Delete the link and record the deletion
  $status = autolink_get('LinkMapper')->delete($form_state['values']['lid']);

  drupal_set_message(t('Deleted link definition for keyword <i>%link</i>.', array('%link' => $form_state['values']['keyword'])));
  watchdog('autolink', 'Deleted link definition for keyword <i>%link</i>.', array('%link' => $form_state['values']['keyword']), WATCHDOG_NOTICE);
  $form_state['redirect'] = 'admin/config/content/autolink';

  return;
}

/**
 * Merges AHAH form values into a readable $link array in preparation for database insert.
 *
 * @return
 *   Merged form values that can be processed for validation or inserted into the database.
 */
function _autolink_merge_link_form_values($form_values) {
  $link = $form_values;
  if (isset($form_values['link_wrapper']['definition'])) {
    $link += $form_values['link_wrapper']['definition'];
  }
  return $link;
}
