<?php

/**
 * @file
 * Defines abstract form classes for building link type forms with the Autolink API.
 */

/**
 * Defines the link type form factory class.
 */
class AutolinkFormFactory {
  private static $instance = array();

  /**
   * Returns an instance of the form definition class for a specific link type.
   * Form classes are stored in a static array for future retrieval.
   */
  static function getInstance($selection) {
    if (!isset(self::$instance[$selection])) {
      $parts = explode('-', $selection);
      if (isset($parts[1])) {
        $type = $parts[0];
        $method = $parts[1];
        $handler = new AutolinkErrorHandler();
        try {
          $info = autolink_get('LinkTypeInfo')->getLinkTypeInfo($type);
          // Include the class file defined by link type info.
          $config = $info['config'][$method];
          $module = !empty($config['module']) ? $config['module'] : $info['module'];
          Autolink::includeFile($module, $config['file']);
          $class = $config['class'];
          if (!class_exists($class)) {
            throw new AutolinkException(AutolinkException::ClassFailed, $class);
          }
          else {
            self::$instance[$selection] = new $class($type, $selection);
          }
        }
        catch (AutolinkException $e) {
          $e->logMessage($handler);
        }
        $handler->displayAll();
      }
    }
    return self::$instance[$selection];
  }
}

/**
 * Defines the abstract AutolinkLinkForm class from which link type specific forms
 * are derived.
 */
abstract class AutolinkLinkForm {
  protected $_type = NULL;
  protected $_selection = NULL;
  protected $config;

  function __construct($type, $selection) {
    $this->_type = $type;
    $this->_selection = $selection;
    $this->config = autolink_get('config');
  }

  function __destruct() {
    $this->_type = NULL;
    $this->_selection = NULL;
  }

  /**
   * Execute callbacks on child classes.
   */
  function execute($method, array $arg) {
    try {
      // We don't want errors on validate calls because that method is optional.
      if ($method != 'validate' && !method_exists($this, $method)) {
        throw new AutolinkException(AutolinkException::FormFailed, $method, $this->_selection, $this->_type);
      }
      else {
        return $this->$method($arg);
      }
    }
    catch (AutolinkException $e) {
      $e->displayMessage();
    }
  }

  /**
   * Processes and saves form values.
   */
  function save(array $form_values) {
    $data = array(
      'group_type' => $form_values['group'],
      'type' => $form_values['type'],
      'method' => $form_values['method'],
    );
    if (isset($form_values['lid'])) {
      $data['lid'] = $form_values['lid'];
    }
    if (isset($form_values['type'])) {
      $data['type'] = $form_values['type'];
    }
    elseif (isset($form_values['method'])) {
      $parts = explode('-', $form_values['method']);
      $data['type'] = $parts[0];
    }
    $data['keyword'] = $this->execute('setKeyword', $form_values);
    $data['path'] = $this->execute('setPath', $form_values);
    $data['query'] = $this->execute('setQuery', $form_values);
    $data['extra'] = method_exists($this, 'setExtra') ? $this->setExtra($form_values) : array();
    autolink_get('link', $data)->save();
  }

  /**
   * The form() method supplies the main link form. This method must be defined
   * by child classes. The $settings variable should be used to provide default
   * values for the form elements. This contains the data that was stored in
   * the database for edit forms.
   *
   * @return
   *   A standard $form array of elements to add to the link add/edit form.
   */
  abstract public function form($settings);

  /**
   * The validate method is optional. It is passed an array of form values.
   */
  function validate($form_values) {}

  /**
   * Sets the primary keyword for the link.
   */
  abstract public function setKeyword($form_values);

  /**
   * Builds the path for the link based on form values.
   */
  abstract public function setPath($form_values);

  /**
   * Sets the query section of the path based on form values. Optional.
   */
  public function setQuery($form_values) {}

  /**
   * Defines extra data to be stored in the database. Optional.
   */
  public function setExtra($form_values) {}
}
