<?php

/**
 * @file
 * Administrative forms for autolink_ module.
 */

/**
 * Settings form for autolink.
 *
 * @ingroup forms
 * @see autolink_form_settings_submit()
 */
function autolink_form_settings($form, &$form_state, $edit = array()) {
  return autolink_get('config')->setForm()->execute('form', $form_state, $edit);
}

/**
 * Validation function for settings form.
 */
function autolink_form_settings_validate($form, &$form_state) {
  return autolink_get('config')->setForm()->execute('validate', $form, $form_state);
}

/**
 * Submit handler for Autolink settings form.
 *
 * This handler sets miltiple value field values by creating individual variables
 * for each value. This makes node and link type settings much easier to work with
 * on other forms and in processing functions and is cleaner than using the
 * 'array_filter' field.
 */
function autolink_form_settings_submit($form, &$form_state) {
  autolink_get('config')->setForm()->execute('submit', $form, $form_state);
}

function autolink_admin_links($callback_arg = '') {
  Autolink::loadFiles(array('operations'));
  if (!empty($_POST['links']) && isset($_POST['operation']) && ($_POST['operation'] == 'delete')) {
    $build['autolink_link_multiple_delete_confirm'] = drupal_get_form('autolink_link_multiple_delete_confirm');
  }
  else {
    $build['autolink_overview_links'] = drupal_get_form('autolink_overview_links');
  }
  return $build;
}

/**
 * Form builder for the groups/links overview page.
 */
function autolink_overview_links($form, $form_state = array()) {
  drupal_add_css(drupal_get_path('module', 'autolink') . '/autolink.css');
  $links = array();
  $result = db_query('SELECT * FROM {autolink_link}');
  foreach ($result as $link) {
    $links[$link->lid] = $link;
  }

  $groups = array();
  $result = db_query('SELECT * FROM {autolink_group}');
  foreach ($result as $group) {
    $types = array();
    $node_result = db_query('SELECT t.* FROM {autolink_group_node_type} g LEFT JOIN {node_type} t ON g.node_type = t.type');
    foreach ($node_result as $type) {
      $types[$type->type] = $type->name;
    }
    $group->node_types = $types;
    $groups[$group->type] = $group;
  }

  $form['links'] = array('#tree' => TRUE);
  foreach ($links as $lid => $link) {
    $form['links'][$link->group_type][$link->lid] = array('#tree' => TRUE);
    $form['links'][$link->group_type][$link->lid]['select'] = array('#type' => 'checkbox', '#title' => t('Select'));
    $form['links'][$link->group_type][$link->lid]['type'] = array('#markup' => $link->type);
    $form['links'][$link->group_type][$link->lid]['keyword'] = array('#markup' => $link->keyword);
    $form['links'][$link->group_type][$link->lid]['destination'] = array('#markup' => drupal_lookup_path('alias', $link->path) ? drupal_lookup_path('alias') : $link->path);
    $form['links'][$link->group_type][$link->lid]['operations'][] = array('data' => array('#type' => 'link', '#title' => t('edit '), '#href' => "admin/config/content/autolink/$link->group_type/edit/link/$link->lid"));
    $form['links'][$link->group_type][$link->lid]['operations'][] = array('data' => array('#type' => 'link', '#title' => t(' delete'), '#href' => "admin/config/content/autolink/$link->group_type/delete/link/$link->lid"));
  }

  foreach ($groups as $group) {
    if (!isset($form['links'][$group->type])) {
      $form['links'][$group->type] = array();
    }
  }

  foreach (element_children($form['links']) as $group) {
    // Create a fieldset with the group description, node types, and links.
    $description = '<div class="autolink-group-info">';
    $description .= '<div class="autolink-group-description">' .  $groups[$group]->description . '</div>';
    $description .= '<div class="autolink-group-node-types">' . implode(', ', $groups[$group]->node_types) . '</div>';
    $description .= '<div class="autolink-group-links">';
    $description .= '<ul><li>' . l(t('edit group'), 'admin/config/content/autolink/edit/group/' . $groups[$group]->type) . '</li>';
    $description .= '<li>' . l(t('add link'), 'admin/config/content/autolink/' . $groups[$group]->type . '/add/link') . '</li></ul>';
    $description .= '</div></div>';

    $form['links'][$group] += array(
      '#type' => 'fieldset',
      '#title' => t($groups[$group]->name),
      '#description' => $description,
      '#collapsible' => TRUE,
      '#theme' => 'autolink_fieldset',
      '#header' => array(
        array('data' => t('Type')),
        array('data' => t('Keyword')),
        array('data' => t('Destination')),
        array('data' => t('Operations'), 'colspan' => 2),
      ),
    );
  }

  uasort($form['links'], 'element_sort_by_title');

  return $form;
}

/**
 * Returns HTML for the modules form.
 *
 * @param $variables
 *   An associative array containing:
 *   - form: A render element representing the form.
 *
 * @ingroup themeable
 */
function theme_autolink_fieldset($variables) {
  $form = $variables['form'];

  // Individual table headers.
  $rows = array();
  foreach (element_children($form) as $key) {
    $link = $form[$key];
    $row = array();
    unset($link['select']['#title']);
    $row[] = array('data' => drupal_render($link['type']));
    $row[] = array('data' => drupal_render($link['keyword']));
    $row[] = array('data' => drupal_render($link['destination']));
    $row[] = array('data' => drupal_render($link['operations']));
    $rows[] = $row;
  }

  return theme('table', array('header' => $form['#header'], 'rows' => $rows));
}

/**
 * Display form for adding and editing link groups.
 *
 * @ingroup forms
 * @see autolink_form_group_submit()
 */
function autolink_form_group($form, &$form_state, $edit = array()) {
  if (!isset($form_state['group'])) {
    $group = is_object($edit) ? $edit : (object) $edit;
    $defaults = array(
      'status' => 1,
      'name' => '',
      'type' => '',
      'description' => '',
      'types' => array(),
    );
    foreach ($defaults as $key => $value) {
      if (!isset($group->$key)) {
        $group->$key = $value;
      }
    }
    $form_state['group'] = $group;
  }
  else {
    $group = $form_state['group'];
  }

  $form['#group'] = $form_state['group'];

  // Check whether we need a deletion confirmation form.
  if (isset($form_state['confirm_delete']) && isset($form_state['values']['type'])) {
    return autolink_group_confirm_delete($form, $form_state, $form_state['values']['type']);
  }

  $form['identification'] = array(
    '#type' => 'fieldset',
    '#title' => t('Identification'),
    '#collapsible' => TRUE,
  );
  $form['identification']['status'] = array(
    '#type' => 'select',
    '#title' => t('Status'),
    '#default_value' => $group->status,
    '#options' => array(0 => t('Disabled'), 1 => t('Active')),
  );
  $form['identification']['name'] = array(
    '#type' => 'textfield',
    '#title' => t('Group name'),
    '#default_value' => $group->name,
    '#maxlength' => 255,
    '#description' => t('The human-readable name for this link group.'),
    '#required' => TRUE,
  );
  $form['identification']['description'] = array(
    '#type' => 'textarea',
    '#title' => t('Description'),
    '#default_value' => $group->description,
    '#description' => t('Description of the link group.'),
  );
  $form['identification']['nodes'] = array(
    '#type' => 'fieldset',
    '#title' => t('Node types'),
    '#collapsible' => TRUE,
  );
  $form['identification']['nodes']['types'] = array(
    '#type' => 'checkboxes',
    '#title' => t('Node types'),
    '#options' => node_type_get_names(),
    '#default_value' => $group->types,
  );

  $form['actions'] = array('#type' => 'actions');
  $form['actions']['submit'] = array('#type' => 'submit', '#value' => t('Save'));
  if (isset($group->type)) {
    $form['actions']['delete'] = array('#type' => 'submit', '#value' => t('Delete'));
    $form['type'] = array('#type' => 'value', '#value' => $group->type);
  }

  if (!isset($group->type)) {
    $form['#validate'][] = 'autolink_form_group_duplicate_validate';
  }

  return $form;
}

/**
 * Validate function for add link group form.
 */
function autolink_form_group_duplicate_validate($form, &$form_state) {
  $group = $form_state['values']['name'];
  $result = db_query("SELECT * FROM {autolink_group} WHERE name = :group", array(':group' => $group))->fetchObject();
  if ($result) {
    form_set_error('name', t('A link group with the name %name already exists.', array('%name' => $group)));
  }
}

/**
 * Submit function for add link group form.
 */
function autolink_form_group_submit($form, &$form_state) {
  if ($form_state['clicked_button']['#value'] == t('Delete')) {
    // Rebuild the form to confirm group deletion.
    $form_state['rebuild'] = TRUE;
    $form_state['confirm_delete'] = TRUE;
    return;
  }

  $group = $form_state['values'];
  switch (autolink_group_save($group)) {
    case SAVED_NEW:
      drupal_set_message(t('Created new link group %name.', array('%name' => $group->name)));
      watchdog('autolink', 'Created new link group %name.', array('%name' => $group->name), WATCHDOG_NOTICE, l(t('edit'), 'admin/config/content/autolink/' . $group->type . '/edit'));
      break;

    case SAVED_UPDATED:
      drupal_set_message(t('Updated link group %name.', array('%name' => $group->name)));
      watchdog('autolink', 'Updated link group %name.', array('%name' => $group->name), WATCHDOG_NOTICE, l(t('edit'), 'admin/config/content/autolink/' . $group->type . '/edit'));
      break;
  }

  $form_state['values']['type'] = $group->type;
  $form_state['type'] = $group->type;
  $form_state['redirect'] = 'admin/config/content/autolink';
}

/**
 * Page to edit a link group.
 */
function autolink_admin_group_edit($group) {
  if ((isset($_POST['op']) && $_POST['op'] == t('Delete')) || isset($_POST['confirm'])) {
    return drupal_get_form('autolink_group_confirm_delete', $group->type);
  }
  return drupal_get_form('autolink_form_group', (array)$group);
}

/**
 * Form builder for the link group delete confirmation form.
 *
 * @ingroup forms
 * @see autolink_group_confirm_delete_submit()
 */
function autolink_group_confirm_delete($form, &$form_state, $type) {
  $group = autolink_group_load($type);

  $form['type'] = array('#type' => 'value', '#value' => $type);

  $form['#group'] = $group;
  $form['#id'] = 'autolink_group_confirm_delete';
  $form['type'] = array('#type' => 'value', '#value' => 'group');
  $form['name'] = array('#type' => 'value', '#value' => $group->name);
  $form['#submit'] = array('autolink_group_confirm_delete_submit');
  return confirm_form($form,
    t('Are you sure you want to delete the link group %title?',
    array('%title' => $group->name)),
    'admin/config/content/autolink',
    t('Deleting a link group will delete all the links in it. This action cannot be undone.'),
    t('Delete'),
    t('Cancel'));
}

/**
 * Submit handler to delete a link group after confirmation.
 *
 * @see autolink_group_confirm_delete()
 */
function autolink_group_confirm_delete_submit($form, &$form_state) {
  $status = autolink_group_delete($form_state['values']['type']);
  drupal_set_message(t('Deleted link group %name.', array('%name' => $form_state['values']['name'])));
  watchdog('autolink', 'Deleted link group %name.', array('%name' => $form_state['values']['name']), WATCHDOG_NOTICE);
  $form_state['redirect'] = 'admin/config/content/autolink';
  cache_clear_all();
  return;
}

/**
 * Form for adding or editing a link definition.
 *
 * This function uses the AHAH Helper module to build a dynamic link definition
 * form. When the AHAH submit function is called and the form is rebuilt this
 * function builds link type specific form elements based on data retrieved with
 * the link type that has been selected.
 */
function autolink_form_link($form, &$form_state, $group, $settings = array()) {
  // Load JS and CSS files for hiding the 'Select' button for AJAX enabled forms.
  drupal_add_js(drupal_get_path('module', 'autolink') . '/autolink.js', 'file');
  drupal_add_css(drupal_get_path('module', 'autolink') . '/autolink.css');

  $form = array();
  $config = autolink_get('Config');
  $form['group'] = array('#type' => 'value', '#value' => $group->type);

  if (!isset($form_state['values']['method'])) {
    $method = isset($settings['method']) ? $settings['method'] : 'select';
  }
  else {
    $method = $form_state['values']['method'];
  }

  // Prepare link types for the form.
  $method_options = array('select' => t('--Select a link type--'));
  $methods = $config->getFormMethods();

  if (!empty($methods)) {
    $method_options += $methods;
  }
  else {
    drupal_set_message(t('There are currently no definable link types enabled. Link types may be enabled by visiting the plugin settings page.'), 'warning');
  }

  // If there is only one link type then set that as the default instead of 'select'.
  $form['method'] = array(
    '#type' => 'select',
    '#title' => t('Link type'),
    '#options' => $method_options,
    '#default_value' => $method,
    '#ajax' => array(
      'event' => 'change',
      'wrapper' => 'autolink-wrapper',
      'callback' => 'autolink_build_link_form',
      'method' => 'replace',
    ),
  );

  $form['method_submit'] = array(
    '#type' => 'submit',
    '#value' => t('Select'),
    '#attributes' => array('class' => array('select-button')),
  );

  $form['link_wrapper'] = array(
    '#prefix' => '<div id="autolink-wrapper">',
    '#suffix' => '</div>',
  );

  // Only if a link type has been selected do we show the link definition form.
  if (!empty($method) && $method != 'select') {
    $parts = explode('-', $method);
    $link_type = $parts[0];
    $form['type'] = array('#type' => 'value', '#value' => $link_type);

    $form['link_wrapper']['definition'] = array(
      '#type' => 'fieldset',
      '#title' => t('Definition'),
      '#collapsible' => TRUE,
    );

    $type_form = AutolinkFormFactory::getInstance($method);
    $form['link_wrapper']['definition'] += $type_form->execute('form', $settings);
  }

  $form['submit'] = array('#type' => 'submit', '#value' => t('Save'));
  $form['group_type'] = array('#type' => 'value', '#value' => $group->type);
  if (isset($settings['lid'])) {
    $form['delete'] = array('#type' => 'submit', '#value' => t('Delete'));
    $form['lid'] = array('#type' => 'value', '#value' => $settings['lid']);
  }
  return $form;
}

/**
 * AJAX callback for the link add/edit form.
 */
function autolink_build_link_form($form, $form_state) {
  return $form['link_wrapper'];
}

/**
 * Validation handler for the link edit form.
 *
 * Here we ensure the link does not contain inappropriate characters. Also, since
 * we are using disabled and hidden fields, we have to do extra form validation
 * here because we cannot use the 'required' => TRUE property on a disabled field.
 *
 * @see autolink_form_link()
 */
function autolink_form_link_validate($form, &$form_state) {
  $values = _autolink_merge_link_form_values($form_state['values']);
  AutolinkFormFactory::getInstance($values['method'])->execute('validate', $values);
}

/**
 * Final submit function for the link form.
 */
function autolink_form_link_submit($form, &$form_state) {
  $values = _autolink_merge_link_form_values($form_state['values']);
  $object = AutolinkFormFactory::getInstance($values['method']);

  switch ($object->execute('save', $values)) {
    case SAVED_NEW:
      drupal_set_message(t('Created new <i>%link_type</i> type link for keyword <i>%keyword</i>.', array('%link_type' => $values['type'], '%keyword' => $values['keyword'])));
      watchdog('autolink', 'Created new link for keyword <i>%keyword</i>.', array('%keyword' => $values['keyword']), WATCHDOG_NOTICE, l(t('edit'), 'admin/content/autolink/' . $values['group_type'] . '/edit/link/' . $values['lid']));
      break;
    case SAVED_UPDATED:
      drupal_set_message(t('Updated <i>%link_type</i> type link for keyword <i>%keyword</i>.', array('%link_type' => $form_state['values']['link_wrapper']['type'], '%keyword' => $values['stored_keyword'])));
      watchdog('autolink', 'Updated link for keyword <i>%keyword</i>.', array('%keyword' => $values['stored_keyword']), WATCHDOG_NOTICE, l(t('edit'), 'admin/content/autolink/'. $values['group_type'] . '/edit/link/' . $link['lid']));
      break;
  }

  $form_state['lid'] = isset($form_state['values']['lid']) ? $form_state['values']['lid'] : '';
  drupal_goto('admin/config/content/autolink/' . $values['group_type']);

  return;
}

/**
 * Page to add a link.
 */
function autolink_add_link_page($group) {
  drupal_set_title(t("Add link definition to $group->name"));
  return drupal_get_form('autolink_form_link', $group);
}

/**
 * Page to edit a link.
 *
 * @param $link
 *   A link object.
 *
 * @see autolink_form_link()
 * @see autolink_confirm_delete()
 */
function autolink_admin_link_edit($group, $lid) {
  if ((isset($_POST['op']) && $_POST['op'] == t('Delete')) || isset($_POST['confirm'])) {
    return drupal_get_form('autolink_confirm_link_delete', $lid);
  }
  $link = autolink_get('LinkMapper')->load($lid);
  return drupal_get_form('autolink_form_link', $group, (array)$link);
}

/**
 * Menu callback for deleting a link.
 */
function autolink_admin_link_delete($group, $lid) {
  return drupal_get_form('autolink_confirm_link_delete', $lid);
}

/**
 * Form builder for the link delete confirmation form.
 *
 * @param $lid
 *   The ID of a link.
 *
 * @ingroup forms
 * @see autolink_confirm_link_delete_submit()
 */
function autolink_confirm_link_delete($form, &$form_state, $lid) {
  $data = autolink_get('LinkMapper')->load($lid);

  $form['type'] = array(
    '#type' => 'value',
    '#value' => 'link',
  );
  $form['lid'] = array(
    '#type' => 'value',
    '#value' => $data['lid'],
  );
  $form['keyword'] = array(
    '#type' => 'value',
    '#value' => $data['keyword'],
  );
  return confirm_form($form,
                  t('Are you sure you want to delete the link %keyword?',
                  array('%keyword' => $data['keyword'])),
                  'admin/config/content/autolink',
                  t('This action cannot be undone.'),
                  t('Delete'),
                  t('Cancel'));
}

/**
 * Submit handler to delete a link after confirmation.
 *
 * @see autolink_confirm_link_delete()
 */
function autolink_confirm_link_delete_submit($form, &$form_state) {
  // Delete the link and record the deletion
  $status = autolink_get('LinkMapper')->delete($form_state['values']['lid']);

  drupal_set_message(t('Deleted link definition for keyword <i>%link</i>.', array('%link' => $form_state['values']['keyword'])));
  watchdog('autolink', 'Deleted link definition for keyword <i>%link</i>.', array('%link' => $form_state['values']['keyword']), WATCHDOG_NOTICE);
  $form_state['redirect'] = 'admin/config/content/autolink';

  return;
}

/**
 * Merges AHAH form values into a readable $link array in preparation for database insert.
 *
 * @return
 *   Merged form values that can be processed for validation or inserted into the database.
 */
function _autolink_merge_link_form_values($form_values) {
  $link = $form_values;
  if (isset($form_values['link_wrapper']['definition'])) {
    $link += $form_values['link_wrapper']['definition'];
  }
  return $link;
}
