<?php

/**
 * @file
 * Defines the primary information gathering classes for processors,
 * plugins, and link types.
 */

/**
 * Defines the information management interface.
 */
interface AutolinkInfoInterface {
  public static function resetCache();
  public function getInfo($type, $name = '');
  public function includeInfo($type, $callback = NULL);
}

/**
 * Information gathering class.
 */
class AutolinkInfo implements AutolinkInfoInterface {
  private static $_info = array();
  private static $_included = array();
  const CACHE_NAME = 'cache_autolink';
  protected $mode;

  private function __construct() {
    $this->mode = variable_get('autolink_development_mode', 0);
  }

  /**
   * Resets the info cache.
   */
  public static function resetCache() {
    self::$_info = array();
    cache_clear_all(NULL, self::CACHE_NAME);
  }

  /**
   * Retrieves plugin information from all modules.
   *
   * Information retrieval is based on the type on information
   * requested. For each module, module_name.autolink.inc is first
   * included if it is found. After retrieving information, modules
   * are allowed the opportunity to alter it with drupal_alter().
   *
   * @param $type
   *   The type of info being retrieved. This can be either plugin or link type info.
   * @param $name
   *   A specific key to return. This parameter is optional.
   */
  function getInfo($type, $name = '') {
    if (!isset(self::$_info[$type]) || empty(self::$_info[$type])) {
      // All autolink hooks should be found in .autolink.inc if not in .module files.
      // Module names are recorded with hook return data for reference in including.
      $hook = 'autolink_' . $type . '_info';
      $this->includeInfo('autolink');
      self::$_info[$type] = array();
      foreach (module_implements($hook) as $module) {
        foreach (module_invoke($module, $hook) as $key => $value) {
          $value['name'] = $key;
          $value['module'] = isset($value['module']) ? $value['module'] : $module;
          self::$_info[$type][$key] = $value;
        }
      }
      drupal_alter($hook, self::$_info[$type]);
    }
  
    // Returns the data about a specific plugin.
    if ($name) {
      return self::$_info[$type][$name];
    }
    return !empty($name) ? self::$_info[$type][$name] : self::$_info[$type];
  }

  /**
   * Includes Autolink include files from external modules.
   *
   * @param $type
   *   A string identifying the type of file to include.
   * @param $callback = NULL
   *   An optional callback which will provide the filenames
   *   to include. Files provided via this method are cached.
   */
  function includeInfo($type, $callback = NULL) {
    if (!isset(self::$_included[$type])) {
      self::$_included[$type] = TRUE;
      if ($cache = cache_get('include_' . $type, self::CACHE_NAME)) {
        $files = $cache->data;
      }
      else {
        if ($type == 'autolink') {
          $files = $this->getAutolinkFiles();
        }
        else {
          $files = $this->getClassFiles($type, $callback);
          cache_set('include_' . $type, $files, self::CACHE_NAME);
        }
      }
  
      foreach ($files as $file) {
        include_once($file);
      }
    }
  }

  /**
   * Checks for files with the .autolink.inc extension.
   */
  private function getAutolinkFiles() {
    $files = array();
    $handler = new AutolinkErrorHandler();
    // Autolink include files do not have definable paths. .autolink.inc only.
    foreach (module_list() as $module) {
      $file = "$module.autolink.inc";
      $this->addFile($files, $module, $file, $handler);
    }
    return $files;
  }

  /**
   * Checks for the existence of files defined in autolink hooks.
   *
   * @param $type
   *   The type of data being included.
   * @param $callback
   *   An optional callback to use instead of getInfo(). This allows
   *   functions to further specify the data to load.
   */
  private function getClassFiles() {
    $files = array();
    $handler = new AutolinkErrorHandler();
    // All other files are included based on the 'file' property set in autolink hooks.
    $autolink_info = !empty($callback) ? $callback() : $this->getInfo($type);
    foreach ($autolink_info as $key => $info) {
      $this->addFile($files, $info['module'], $info['file'], $handler);
    }
    $handler->displayAll();
    return $files;
  }

  /**
   * Adds a file to an array if it exists.
   */
  private function addFile(&$files, $module, $file, $handler = NULL) {
    $path = drupal_get_path('module', $module);
    $file = "$path/" . $file;
    try {
      if (!file_exists($file)) {
        throw new AutolinkException(AutolinkException::IncludeFailed, $file);
      }
      else {
        $files[] = $file;
      }
    }
    catch (AutolinkException $e) {
      $e->logMessage($handler);
    }
  }
}

/**
 * Defines the processor info interface.
 */
interface AutolinkProcessorInfoInterface {
  function getProcessorInfo($processor = NULL);
}

/**
 * Defines the processor info class.
 */
class AutolinkProcessorInfo extends AutolinkInfo implements AutolinkProcessorInfoInterface {
  private $_info = array();

  /**
   * Standard constructor method.
   */
  function __construct() {
    $this->_info = parent::getInfo('processor');
    if ($this->mode == 1 && !empty($this->_info)) {
      $handler = new AutolinkErrorHandler();
      try {
        $this->validate();
      }
      // Log caught messages for display.
      catch (AutolinkException $e) {
        $e->logMessage($handler);
      }
      // Display error messages to the screen (if any).
      $handler->displayAll();
    }
  }

  /**
   * Returns processor information.
   */
  function getProcessorInfo($processor = NULL) {
    if (!empty($processor)) {
      return $this->_info[$processor];
    }
    else {
      return $this->_info;
    }
  }

  /**
   * Validates processor information.
   */
  function validate() {
    foreach ($this->_info as $info) {
      // Ensure that the plugin file path is not a full url.
      if (url_is_external($info['file'])) {
        throw new AutolinkException(AutolinkException::PluginInfo, "Plugin file must be defined relative to the base folder of the defining module.");
      }
    }
  }
}

/**
 * Defines the Autolink plugin information interface.
 */
interface AutolinkPluginInfoInterface {
  public function getAll();
  public function getCurrent();
  public function getPlugin($plugin);
  public function getPluginInfo($field = NULL, $key = NULL, $plugin = NULL);
  public function getLabels();
}

/**
 * Defines the Autolink plugin information class.
 */
class AutolinkPluginInfo extends AutolinkInfo implements AutolinkPluginInfoInterface {
  private $_info = array();
  private $enabledPlugin = NULL;

  function __construct() {
    $this->_info = parent::getInfo('plugin');
    $this->enabledPlugin = variable_get('autolink_plugin', AUTOLINK_DEFAULT_PLUGIN);
    $this->prepare();
    if ($this->mode == 1 && !empty($this->_info[$this->enabledPlugin]) && is_array($this->_info[$this->enabledPlugin])) {
      $handler = new AutolinkErrorHandler();
      try {
        $this->validate();
      }
      // Log caught messages for display.
      catch (AutolinkException $e) {
        $e->logMultiple($handler, array('message', 'file', 'line'));
      }
      // Display error messages to the screen (if any).
      $handler->displayAll();
    }
  }

  /**
   * Sets the enabled plugin for unit testing.
   */
  function setEnabledPlugin($plugin) {
    $this->enabledPlugin = $plugin;
  }

  /**
   * Returns information about all plugins.
   */
  function getAll() {
    return $this->_info;
  }

  /**
   * Returns information about a specific plugin.
   */
  function getPlugin($plugin) {
    return $this->_info[$plugin];
  }

  /**
   * Returns the current plugin information.
   */
  function getCurrent() {
    return isset($this->_info[$this->enabledPlugin]) ? $this->_info[$this->enabledPlugin] : array();
  }

  /**
   * Returns specific plugin information. If no plugin is specified, the currently
   * enabled plugin is used.
   */
  function getPluginInfo($field = NULL, $key = NULL, $plugin = NULL) {
    $plugin = !empty($plugin) ? $plugin : $this->enabledPlugin;
    if (!empty($field) && !empty($key)) {
      return $this->_info[$plugin][$field][$key];
    }
    elseif (!empty($field)) {
      return $this->_info[$plugin][$field];
    }
    else {
      return $this->_info[$plugin];
    }
  }

  /**
   * Returns an array of labels to be used in forms.
   */
  function getLabels() {
    $labels = array();
    foreach ($this->_info as $key => $info) {
      $labels[$key] = $info['label'];
    }
    return $labels;
  }

  /**
   * Sets default values on unset elements of plugin info. Boolean values
   * are converted to Autolink constants for easy reference.
   */
  private function prepare() {
    $plugin = $this->getCurrent();
    // Set defined files to *.autolink.inc if none were specified.
    if ($module = $plugin['module']) {
      if (!isset($plugin['file']) || empty($plugin['file'])) {
        $this->_info[$this->enabledPlugin]['file'] = "$module.autolink.inc";
      }
      if (isset($plugin['config']) && (!isset($plugin['config']['file']) || empty($plugin['config']['file']))) {
        $this->_info[$this->enabledPlugin]['config']['file'] = "$module.autolink.inc";
      }
    }
  }

  /**
   * Defines required fields for plugins.
   */
  private function requirements() {
    return array('label', 'class', 'link types');
  }

  /**
   * Throws errors if plugin information is missing or incorrect.
   * This process is only run on the currently enabled plugin.
   */
  private function validate() {
    $plugin = $this->getCurrent();
    $link_types = autolink_get('LinkTypeInfo')->getLinkTypeInfo();
    // Check to ensure that required fields exist in the plugin info.
    foreach ($this->requirements() as $requirement) {
      if (!array_key_exists($requirement, $plugin) || empty($plugin[$requirement])) {
        throw new AutolinkException(AutolinkException::PluginInfo, "Plugin must define $requirement.");
      }
    }
    // Ensure that the plugin file path is not a full url.
    if (url_is_external($plugin['file'])) {
      throw new AutolinkException(AutolinkException::PluginInfo, "Plugin file must be defined relative to the base folder of the defining module.");
    }
    // If configuration forms are defined, check for a class and file.
    if ($plugin['config']) {
      if (!isset($plugin['config']['class'])) {
        throw new AutolinkException(AutolinkException::PluginInfo, "Plugin configuration must define a configuration form class.");
      }
      if (!isset($plugin['config']['file'])) {
        throw new AutolinkException(AutolinkException::PluginInfo, "Plugin configuration must identify the file that contains configuration class.");
      }
      elseif (url_is_external($plugin['config']['file'])) {
        throw new AutolinkException(AutolinkException::PluginInfo, "Plugin configuration file must be defined relative to the base folder of the defining module.");
      }
    }
    // Ensure that all link types that are specified are actually available.
    if (is_array($plugin['link types'])) {
      foreach ($plugin['link types'] as $type) {
        if (!array_key_exists($type, $link_types)) {
          throw new AutolinkException(AutolinkException::PluginInfo, "Cannot locate link type data for $type as defined by the {$plugin['label']} plugin.");
        }
      }
    }
    if (!isset($plugin['method']) || empty($plugin['method'])) {
      throw new AutolinkException(AutolinkException::PluginInfo, "The plugin must specify a processing method.");
    }
    elseif ($plugin['method'] == 'filter' && !module_hook($plugin['module'], 'autolink_filter')) {
      throw new AutolinkException(AutolinkException::PluginInfo, "Plugins using the filter method must implement hook_autolink_filter().");
    }
  }
}

/**
 * Defines the link type information interface.
 */
interface AutolinkLinkTypeInfoInterface {
  public function getAll();
  public function getType($type);
  public function getLinkTypeInfo($type = NULL, $field = NULL);
  public function isIdentifiable($type);
  public function getIdentifiable($field = NULL);
  public function isDefinable($type);
  public function getDefinable($field = NULL);
}

/**
 * Defines the link type information class.
 */
class AutolinkLinkTypeInfo extends AutolinkInfo implements AutolinkLinkTypeInfoInterface {
  private static $instance;
  private $_info = array();

  /**
   * Object constructor.
   *
   * Link type info objects are constructed by first retrieving the link
   * type info from AutolinkInfo class, converting boolean values with
   * the prepare() method, and validating the information with validate().
   * The validation ensures that information is properly formatted.
   */
  function __construct() {
    $this->_info = parent::getInfo('link_type');
    $this->prepare();
    if ($this->mode == 1) {
      $handler = new AutolinkErrorHandler();
      try {
        $this->validate();
      }
      catch (AutolinkException $e) {
        $e->logMultiple($handler, array('message', 'file', 'line'));
      }
      $handler->displayAll();
    }
  }

  /**
   * Returns all link type information.
   */
  function getAll() {
    return $this->_info;
  }

  /**
   * Returns information about a specific link type.
   */
  function getType($type) {
    return $this->_info[$type];
  }

  /**
   * Returns information on specific fields of specific link type, or all
   * information about all link types if no fields are defined.
   */
  function getLinkTypeInfo($type = NULL, $field = NULL) {
    if (!empty($type) && !empty($field)) {
      return $this->_info[$type][$field];
    }
    elseif (!empty($type)) {
      return $this->_info[$type];
    }
    else {
      return $this->_info;
    }
  }

  /**
   * Returns information for active link types. The activity of link types
   * is determined by the currently enabled plugin.
   */
  function getActive() {
    $plugin = autolink_get('PluginInfo')->getCurrent();
    $active = array();
    foreach ($this->_info as $key => $value) {
      if (in_array($key, $plugin['link types'])) {
        $active[$key] = $value;
      }
    }
    return $active;
  }

  /**
   * Returns abstract link types info (link types that are not definable).
   */
  function getAbstract($plugin = NULL) {
    $abstracts = array();
    foreach ($this->_info as $type => $info) {
      if (!$this->isDefinable($type)) {
        $abstracts[$type] = $info;
      }
    }
    return $abstracts;
  }

  /**
   * Returns a boolean value representing whether a link type is identifiable.
   */
  function isIdentifiable($type) {
    return $this->_info[$type]['identifiable'] == AUTOLINK_ENABLED;
  }

  /**
   * Returns a boolean value representing whether a link type is definable.
   */
  function isDefinable($type) {
    return $this->_info[$type]['definable'] == AUTOLINK_ENABLED;
  }

  /**
   * Returns an array of link types that are identifiable. If a $field is
   * specified, that field will be return in values instead of all information.
   *
   * @param $field
   *   An optional array key to use as the value of the returned array. This
   *   allows the function to return identifiable labels instead of full info.
   *
   * @return
   *   An associative array of identifiable link type data. If a field is
   *   specified, that field will be returned in values instead of full info.
   */
  function getIdentifiable($field = NULL) {
    $identifiable = array();
    foreach ($this->_info as $key => $value) {
      if (self::isIdentifiable($key)) {
        $identifiable[$key] = !empty($field) ? $value[$field] : $value;
      }
    }
    return $identifiable;
  }

  /**
   * Returns an array of link type that are definable. If a $field is specified,
   * that field will be returned in values instead of all information. Link types
   * returned are active based on the settings of the currently enabled plugin.
   */
  function getDefinable($field = NULL) {
    $definable = array();
    foreach ($this->getActive() as $key => $value) {
      if (self::isDefinable($key)) {
        $definable[$key] = !empty($field) ? $value[$field] : $value;
      }
    }
    return $definable;
  }

  /**
   * Prepares link type info by setting default values if necessary.
   * Boolean values are converted to Autolink constants for easier identification.
   */
  private function prepare() {
    foreach ($this->_info as $type => $info) {
      if (!isset($info['identifiable']) || $info['identifiable'] == TRUE) {
        $this->_info[$type]['identifiable'] == AUTOLINK_ENABLED;
      }
      elseif ($info['identifiable'] != AUTOLINK_ENABLED) {
        $this->_info[$type]['identifiable'] == AUTOLINK_DISABLED;
      }
      if (!isset($info['definable']) || $info['definable'] == TRUE) {
        $this->_info[$type]['definable'] == AUTOLINK_ENABLED;
      }
      elseif ($info['definable'] != AUTOLINK_ENABLED) {
        $this->_info[$type]['definable'] == AUTOLINK_DISABLED;
      }
      // Set the class file to the default autolink.inc file if none is specified.
      if (!isset($info['file']) && isset($info['module'])) {
        $module = $info['module'];
        $this->_info[$type]['file'] = "$module.autolink.inc";
      }
    }
  }

  /**
   * Checks to verify the structure of link type information.
   */
  private function validate() {
    foreach ($this->_info as $type => $info) {
      // Identifiable link types must provide labels.
      if (!empty($info['identifiable']) && (!isset($info['label']) || empty($info['label']))) {
        throw new AutolinkException(AutolinkException::LinkTypeInfo, "Identifiable link types must define a label in link type $type.");
      }
      // All link types must provide a base class.
      if (!isset($info['class']) || empty($info['class'])) {
        throw new AutolinkException(AutolinkException::LinkTypeInfo, "Link type must define a class in link type $type.");
      }
      // Check if the link type is definable.
      if ($info['definable'] == AUTOLINK_ENABLED) {
        // Definable link types must also be identifiable.
        if ($info['identifiable'] == AUTOLINK_DISABLED) {
          throw new AutolinkException(AutolinkException::LinkTypeInfo, "Definable link types must also be identifiable in link type $type.");
        }
        // Definable link types must provide configuration information.
        if (empty($info['config']) || !isset($info['config'])) {
          throw new AutolinkException(AutolinkException::LinkTypeInfo, "Definable link types must define configuration forms in link type $type.");
        }
        else {
          foreach ($info['config'] as $title => $args) {
            // Check for a configuration file.
            if (!isset($args['file']) || empty($args['file'])) {
              throw new AutolinkException(AutolinkException::LinkTypeInfo, "Link type forms must specify a class location with the key 'file' in link type $type.");
            }
            // Ensure that the given file is not a full url.
            if (url_is_external($args['file'])) {
              throw new AutolinkException(AutolinkException::LinkTypeInfo, "Link type files should be identified relative to the base directory of the calling module's folder in link type $type.");
            }
            // Check for a configuration class.
            if (!isset($args['class'])) {
              throw new AutolinkException(AutolinkException::LinkTypeInfo, "Link type forms must specify a class with the key 'class' in link type $type.");
            }
            // Check for a label.
            if (!isset($args['label'])) {
              throw new AutolinkException(AutolinkException::LinkTypeInfo, "Link type forms must specific a label with the key 'label' in link type $type.");
            }
          }
        }
      }
      // Link types that are not definable must define a provider class.
      elseif (!isset($info['provider'])) {
        throw new AutolinkException(AutolinkException::LinkTypeInfo, "Non-definable link types must define a provider class in link type $type");
      }
    }
  }
}
