<?php

/**
 * @file
 * Administrative forms for autolink_ module.
 */

/**
 * Enables a plugin via a menu callback.
 */
function autolink_plugin_enable($format, $plugin) {
  $config = autolink_config($format)
    ->enable_plugin($plugin)
    ->save();
  drupal_set_message(t('The %plugin plugin has been enabled.', array(
    '%plugin' => $config->plugin_info[$plugin]['label'])));
  drupal_goto("admin/config/content/formats/$format");
}

/**
 * Disables a plugin via a menu callback.
 */
function autolink_plugin_disable($format, $plugin) {
  $config = autolink_config($format)
    ->disable_plugin($plugin)
    ->save();
  drupal_set_message(t('The %plugin plugin has been disabled.', array(
    '%plugin' => $config->plugin_info[$plugin]['label'])));
  drupal_goto("admin/config/content/formats/$format");
}



/**
 * Plugin settings form.
 */
function autolink_form_plugin($form, &$form_state, $format, $plugin_name) {
  // Load include files that were identified in hook_autolink_file_info().
  autolink_load_includes();

  $plugin = autolink_plugin()
    ->init($plugin_name, $format);
  drupal_set_title(t('Configure @plugin plugin', array('@plugin' => $plugin->label)));

  $form = array();
  $form['format'] = array('#type' => 'value', '#value' => $format);
  $form['plugin'] = array('#type' => 'value', '#value' => $plugin->name);
  $form['description'] = array(
    '#type' => 'item',
    '#description' => isset($plugin->description) ? $plugin->description : '',
  );

  // Retrieve the plugin defined settings form.
  $callback = $plugin->get_callback('form');
  if (function_exists($callback)) {
    $callback($form, $plugin->settings, $plugin_name, $format);
  }

  // If the plugin does not define a form then display a message saying such.
  $children = element_children($form);
  if (empty($children)) {
    $form['configuration']['empty'] = array(
      '#value' => t('This plugin has no configuration options.'),
    );
  }
  $form['submit'] = array('#type' => 'submit', '#value' => t('Save configuration'));
  $form['buttons']['reset'] = array(
    '#type' => 'submit',
    '#value' => t('Reset to defaults'),
    '#weight' => 10,
  );
  return $form;
}

/**
 * Validation handler for plugin configuration form.
 */
function autolink_form_plugin_validate($form, &$form_state) {
  $plugin = autolink_plugin()->init($form_state['values']['plugin'], $form_state['values']['format']);
  $callback = $plugin->get_callback('validate');
  if (function_exists($callback)) {
    $callback($form, $form_state, $form_state['values']['plugin'], $form_state['values']['format']);
  }
}

/**
 * Submit handler for the plugin configuration form.
 */
function autolink_form_plugin_submit($form, &$form_state) {
  $plugin = autolink_plugin()->init($form_state['values']['plugin'], $form_state['values']['format']);
  $op = isset($form_state['values']['op']) ? $form_state['values']['op'] : '';
  // Exclude unnecessary elements.
  unset($form_state['values']['submit'], $form_state['values']['reset'], $form_state['values']['form_id'], $form_state['values']['op'], $form_state['values']['form_token'], $form_state['values']['form_build_id']);

  if ($op == t('Reset to defaults')) {
    $plugin->reset();
  }
  else {
    foreach ($form_state['values'] as $key => $value) {
      if (is_array($value)) {
        $value = array_filter($value);
      }
      $plugin->settings[$key] = $value;
    }
  }
  $plugin->save();

  if ($op == t('Reset to defaults')) {
    drupal_set_message(t('The configuration options have been reset to their default values.'));
  }
  else {
    drupal_set_message(t('The configuration options have been saved.'));
  }
  drupal_goto('admin/config/content/formats/' . $form_state['values']['format']);
}

/**
 * Defines the Autolink configuration interface for managing settings.
 */
interface AutolinkConfiguration {
  function get($var, $default = 0);
  function set($var, $value);
  function reset();
  function delete($var);
  function save();
  function setting_enabled($setting);
}

/**
 * Defines the plugin configuration interface.
 */
interface AutolinkPluginActions {
  function enable_plugin($plugin);
  function disable_plugin($plugin);
  function plugin_is_enabled($plugin);
  function enabled_plugins();
}

/**
 * Defines the configuration class for handling Autolink administration.
 */
class AutolinkConfig implements AutolinkConfiguration, AutolinkPluginActions {
  public $settings = array();
  public $plugins = array();
  public $plugin_info = array();
  public $format = -1;

  public function __construct($format) {
    $this->format = $format;
    $this->plugin_info = autolink_get_info('plugin');
    $settings = variable_get('autolink_settings_' . $format, array());
    $this->settings = isset($settings['global']) ? $settings['global'] : array();
    $this->plugins = isset($settings['plugins']) ? $settings['plugins'] : array();
  }

  /**
   * Returns global configuration settings.
   */
  public function __get($property) {
    return isset($this->settings[$property]) ? $this->settings[$property] : FALSE;
  }

  /**
   * Returns a setting or a default value if none is found.
   */
  public function get($var, $default = 0) {
    return isset($this->settings[$var]) ? $this->settings[$var] : $default;
  }

  /**
   * Sets a setting.
   */
  public function set($var, $value) {
    $this->settings[$var] = $value;
  }

  /**
   * Resets Autolink settings to defaults.
   */
  public function reset() {
    $this->settings = array();
  }

  /**
   * Deletes a Drupal variable.
   */
  public function delete($var) {
    unset($this->settings[$var]);
  }

  /**
   * Saves settings for the current plugin.
   */
  public function save() {
    $settings['global'] = $this->settings;
    $settings['plugins'] = $this->plugins;
    variable_set('autolink_settings_' . $this->format, $settings);
    return $this;
  }

  /**
   * Enables a plugin.
   */
  public function enable_plugin($plugin) {
    $this->plugins[$plugin] = TRUE;
    return $this;
  }

  /**
   * Disables a plugin.
   */
  public function disable_plugin($plugin) {
    $this->plugins[$plugin] = FALSE;
    return $this;
  }

  /**
   * Returns whether a plugin is enabled.
   */
  public function plugin_is_enabled($plugin) {
    return !empty($this->plugins[$plugin]);
  }

  /**
   * Returns all enabled plugins.
   */
  public function enabled_plugins() {
    return array_filter($this->plugins);
  }

  /**
   * Returns whether a setting is enabled.
   */
  public function setting_enabled($setting) {
    return !empty($this->settings[$setting]);
  }

}
