<?php

/**
 * @file
 * Autolink text processors.
 */

/**
 * Define the cache interface.
 */
interface AutolinkPluginCache {
  public function set_items($id, $items);
  public function get_items($id);
  public function clear_items($id);
}

/**
 * Cache object class handles caching within processors. During preprocessing,
 * the cache object is passed to plugin callbacks, allowing them to easily
 * cache data in the Autolink cache.
 */
class AutolinkCache implements AutolinkPluginCache {
  var $plugin;
  var $format;

  public function __construct($plugin, $format) {
    $this->plugin = $plugin;
    $this->format = $format;
  }

  /**
   * Caches item arrays for specific plugins. This should be used by plugins
   * within the scope of the preprocess callback.
   */
  public function set_items($id, $items, $expire = CACHE_PERMANENT) {
    if (!empty($items) && is_array($items)) {
      cache_set($this->plugin . '_' . $this->format . '_items_' . $id, $items, AUTOLINK_CACHE, $expire);
    }
  }

  /**
   * Returns cached items from the Autolink cache. This may be used by plugins
   * within the scope of the preprocess callback.
   */
  public function get_items($id) {
    return cache_get($this->plugin . '_' . $this->format . '_items_' . $id, AUTOLINK_CACHE);
  }

  /**
   * Clears the items cache for a specific plugin.
   */
  public function clear_items($id) {
    cache_clear_all($this->plugin . '_' . $this->format . '_items_' . $id, AUTOLINK_CACHE);
  }

}

/**
 * Defines the Autolink plugin interface.
 */
interface AutolinkPluginObject {
  public function init($plugin, $format);
  public function get_info();
  public function save();
  public function get($var, $default = 0);
  public function set($var, $value);
}

/**
 * Plugins handle actual processing of text and generation of links.
 */
class AutolinkPlugin implements AutolinkPluginObject {
  protected $config;
  protected $info = array();
  public $cache;
  public $format;
  public $name = NULL;
  public $module = NULL;
  public $label = '';
  public $description = '';
  public $processor = NULL;
  public $settings = array();

  public function __construct() {}

  /**
   * Initializes a plugin instance. Properties are populated from plugin info.
   *
   * @param $plugin
   *   The machine-name of the plugin to initialize.
   * @param $format
   *   The machine-name of the format related to this plugin instance.
   * @return
   *   An initialized AutolinkPlugin object.
   */
  public function init($plugin, $format) {
    $this->config = autolink_config($format);
    $this->format = $format;
    if ($plugin_info = $this->config->plugin_info[$plugin]) {
      $this->name = $plugin;
      $this->module = $plugin_info['module'];
      $this->info = $plugin_info;
      $this->cache = new AutolinkCache($plugin, $format);
      foreach ($plugin_info as $property => $value) {
        $this->{$property} = $value;
      }
      $this->settings = variable_get('autolink_plugin_' . $this->name . '_' . $this->format, array());
      $this->weight = isset($this->settings['weight']) ? $this->settings['weight'] : 0;
    }
    return $this;
  }

  /**
   * Magic setter method.
   */
  public function __set($property, $value) {
    $this->{$property} = $value;
  }

  /**
   * Magic call method for calling functions based on plugin info.
   *
   * This allows plugin modules to define their own callbacks and call
   * it by using $plugin->execute_(callback). We search for 'execute_'
   * in the method and if found strip it and get the proper callback
   * from the plugin info. If no callback was identified then
   * {module}_{plugin}_{callback_name} is used.
   *
   * Since this magic callback is used many times during processing,
   * we statically cache functions that do not exist to prevent having
   * to repeat the check dozens of times in the same process.
   */
  public function __call($method, $args) {
    static $callbacks = array();
    // If the callback exists and is known it will be stored in the static array.
    if (isset($callbacks[$this->name][$method])) {
      // Callbacks stored in the static variable have already been checked.
      if (!empty($callbacks[$this->name][$method])) {
        return call_user_func_array($callbacks[$this->name][$method], $args);
      }
      // The callback was not found previously so return nothing.
      return;
    }
    // We haven't checked for this callback yet, so parse the method and check for it.
    elseif (strstr($method, 'execute_')) {
      $name = substr($method, 8);
      $callback = $this->get_callback($name);
      if (function_exists($callback)) {
        $callbacks[$this->name][$method] = $callback;
        return call_user_func_array($callback, $args);
      }
      // Callbacks that don't exist are noted in the static array as FALSE.
      else {
        $callbacks[$this->name][$method] = FALSE;
      }
    }
  }

  /**
   * Returns a callback string as identified in plugin information.
   */
  public function get_callback($name) {
    return isset($this->info['callbacks'][$name]) ? $this->info['callbacks'][$name] : "{$this->module}_{$this->name}_{$name}";
  }

  /**
   * Returns plugin info.
   */
  public function get_info() {
    return $this->info;
  }

  /**
   * Saves plugin settings.
   */
  public function save() {
    $this->settings['weight'] = $this->weight;
    variable_set('autolink_plugin_' . $this->name . '_' . $this->format, $this->settings);
  }

  /**
   * Returns a plugin setting.
   */
  public function get($var, $default = 0) {
    return isset($this->settings[$var]) ? $this->settings[$var] : $default;
  }

  /**
   * Sets a plugin setting.
   */
  public function set($var, $value) {
    $this->settings[$var] = $value;
  }

  /**
   * Resets the plugin settings.
   */
  public function reset() {
    $this->settings = array();
  }

}
