<?php

/**
 * @file
 * Defines data processors for Autolink module.
 */

/**
 * Defines an interface for processing filtered text.
 */
interface AutolinkFilterProcessor {
  public static function get_processor($processor, AutolinkPlugin $plugin);
  public function add_item($item);
  public function add_items(array $items);
  public function skip();
  public function stop();
}

/**
 * The limits interface defines functions for imposing item limits.
 */
interface AutolinkProcessorLimits {
  public function set_total_limit($limit);
  public function set_item_limit($limit);
}

/**
 * The base processor class uses the singleton method to return processor objects.
 */
abstract class AutolinkProcessor implements AutolinkFilterProcessor, AutolinkProcessorLimits {
  private static $instances;
  var $name;
  protected static $info;
  var $plugin;
  var $items = array();
  var $total_limit = AUTOLINK_DEFAULT_LIMIT;
  var $item_limit = AUTOLINK_DEFAULT_LIMIT;
  var $total = 0;
  var $validate_items = TRUE;
  var $validate_limits = TRUE;
  var $validate_nids = TRUE;
  var $skip = FALSE;
  var $stop = FALSE;

  /**
   * Loads, statically caches, and returns a processor object of a specific type.
   */
  public static function get_processor($processor, AutolinkPlugin $plugin) {
    // Get all info if this is the first initialization.
    if (!isset(self::$info)) {
      self::$info = autolink_get_info('processor');
    }

    // Statically cache and return the identified processor.
    if (!isset(self::$instances[$processor])) {
      self::$instances[$processor] = isset(self::$info[$processor]['class']) ? new self::$info[$processor]['class']($processor) : FALSE;
    }

    // Initialize the processor with the current plugin object.
    if (self::$instances[$processor] instanceof AutolinkProcessor) {
      self::$instances[$processor]->plugin = $plugin;
      self::$instances[$processor]->reset();
    }
    // If the processor info couldn't be found then FALSE will always be returned.
    return self::$instances[$processor];
  }

  /**
   * Private constructor maintains the singleton pattern.
   */
  private function __construct($name) {
    $this->name = $name;
  }

  /**
   * Resets all properties when a new plugin is initialized to reduce
   * code in which the new plugin may have to reset settings.
   */
  private function reset() {
    $this->items = array();
    $this->total = 0;
    $this->total_limit = AUTOLINK_DEFAULT_LIMIT;
    $this->item_limit = AUTOLINK_DEFAULT_LIMIT;
    $this->validate_items = TRUE;
    $this->validate_limits = TRUE;
    $this->validate_nids = TRUE;
    $this->skip = FALSE;
    $this->stop = FALSE;
  }

  /**
   * Halts the processing of text. This method should be used by plugins to
   * prevent the continuation within a plugin for specific content types for example.
   */
  public function skip() {
    $this->skip = TRUE;
  }

  /**
   * Completely stops processing of text.
   */
  public function stop() {
    $this->stop = TRUE;
  }

  /**
   * Adds an item to the items array.
   */
  public function add_item($item) {
    $this->items[] = $item;
  }

  /**
   * Adds an array of items to the items array.
   */
  public function add_items(array $items) {
    $this->items = array_merge($this->items, $items);
  }

  /**
   * Sets a total limit.
   */
  public function set_total_limit($limit) {
    $this->total_limit = $limit;
  }

  /**
   * Sets an item limit.
   */
  public function set_item_limit($limit) {
    $this->item_limit = $limit;
  }

  /**
   * Returns a limit based on plugin and item limits. The returned limit
   * is safe for use in regular expression functions. For no limit, this
   * method returns -1, for example.
   */
  protected function get_item_limit() {
    if ($this->total_limit == 'all' && $this->item_limit == 'all') {
      $limit = -1;
    }
    elseif ($this->total_limit == 'all' && $this->item_limit != 'all') {
      $limit = $item_limit;
    }
    elseif ($this->total_limit != 'all' && $this->item_limit == 'all') {
      $limit = $total_limit;
    }
    if ($this->total_limit != 'all' && $this->item_limit != 'all') {
      if ($this->item_limit > ($this->total_limit - $total)) {
        $limit = $this->total_limit - $this->total;
      }
      else {
        $limit = $this->item_limit;
      }
    }
    return $limit;
  }

  /**
   * Adds a count to the total count.
   */
  protected function add_count($count) {
    $this->total = $this->total + $count;
  }

  /**
   * Returns FALSE if the current loop has reached its limit.
   */
  protected function valid_limit() {
    if ($this->validate_limits && ($this->total_limit !== AUTOLINK_DEFAULT_LIMIT && $this->total >= $this->total_limit)) {
      return FALSE;
    }
    return TRUE;
  }

  /**
   * If an 'nid' is set, returns FALSE if the item's nid equals the node nid.
   */
  protected function valid_nid($node, $nid) {
    if ($this->validate_nids && (isset($item['nid']) && $item['nid'] == $node->nid)) {
      return FALSE;
    }
    return TRUE;
  }

  /**
   * Returns FALSE if an item does not have 'search' and 'replace' values.
   */
  protected function valid_item($item) {
    if ($this->validate_items && (!isset($item['search']) || !isset($item['replace']))) {
      drupal_set_message(t('Autolink has encountered malformed data.'));
      return FALSE;
    }
    return TRUE;
  }

  /**
   * Calls a plugin's replacement callback to retrieve match replacements.
   */
  protected function get_replacement($match, $item = array()) {
    $replacement = $this->plugin->execute_replace($match, $this->plugin->settings, $item);
    if (!empty($replacement)) {
      return $replacement;
    }
    // Return the replace if it's set or just the matched text if not.
    return isset($item['replace']) ? $item['replace'] : $match;
  }

  abstract public function execute(&$text, $node, array $items);

}

/**
 * The default processor class. Performs simple search and replace operations.
 */
class AutolinkKeywordProcessor extends AutolinkProcessor {
  var $case = 0;
  var $prefix = '/';
  var $suffix = '/';

  /**
   * Processes the filter with each item in a loop.
   */
  public function execute(&$text, $node, array $items) {
    foreach ($items as $id => $item) {
      if (!$this->valid_limit()) {
        break;
      }
      elseif ($this->valid_item($item) && $this->valid_nid($node, $item)) {
        $text = preg_replace(
          $this->get_expression($item),
          '$this->get_replacement("$1", $item)',
          $text,
          $this->get_item_limit(),
          $count
        );
        // Track the count of replacements made for plugins that impose limits.
        if (!empty($count)) {
          $this->add_count($count);
        }
      }
    }
  }

  /**
   * Returns a regular expression. If the $link->case property is set,
   * case-sensitivity will be followed. If not it will be insensitive.
   */
  protected function get_expression($item) {
    $expression = $this->prefix . '(' . preg_quote($item['search']) . ')' . $this->suffix;
    $expression .= !empty($this->case) ? 'e' : 'ie';
    return $expression;
  }

}

/**
 * The pattern processor class.
 */
class AutolinkPatternProcessor extends AutolinkProcessor {
  var $expression = '';

  /**
   * Overrides the default execute method, which loops through items.
   * This method instead loops through pattern matches.
   */
  public function execute(&$text, $node, array $items) {
    preg_match_all($this->expression, $text, $matches);
    // Items are checked outside the foreach loop instead of inside because
    // otherwise we would have another conditional on each loop.
    if (!empty($items) && !empty($matches[0])) {
      foreach ($matches[0] as $match) {
        if (!$this->valid_limit()) {
          break;
        }
        $this->loop_items($text, $node, $items, $match);
      }
    }
    elseif (!empty($matches[0])) {
      foreach ($matches[0] as $match) {
        if (!$this->valid_limit()) {
          break;
        }
        $replacement = $this->get_replacement($match);
        $text = str_replace($match, $replacement, $text, $count);
        $this->add_count($count);
      }
    }
  }

  /**
   * Loops matches with items.
   */
  protected function loop_items(&$text, $node, $items, $match) {
    if (!$this->valid_limit()) {
      break;
    }
    else {
      foreach ($items as $id => $item) {
        if ($match == '[' . $item['search'] . ']') {
          $this->check_nid($item);
          $replacement = $this->get_replacement($match, $item);
          $text = str_replace($match, $replacement, $text, $count);
          $this->add_count($count);
        }
      }
    }
  }

}
