<?php

/**
 * @file
 * Administrative forms for the Autolink Default Plugin module.
 */

/**
 * Page to add a link.
 */
function autolink_keyword_add_link_page($plugin, $format) {
  return drupal_get_form('autolink_keyword_form_link', $plugin, $format);
}

/**
 * Page to edit a link.
 */
function autolink_keyword_admin_link_edit($plugin, $format, $link) {
  if ((isset($_POST['op']) && $_POST['op'] == t('Delete')) || isset($_POST['confirm'])) {
    return drupal_get_form('autolink_keyword_confirm_link_delete', $plugin, $format, (array)$link);
  }
  return drupal_get_form('autolink_keyword_form_link', $plugin, $format, (array)$link);
}

/**
 * Page to delete a link.
 */
function autolink_keyword_delete_link_page($plugin, $format, $link) {
  return drupal_get_form('autolink_keyword_confirm_link_delete', $plugin, $format, (array)$link);
}

/**
 * Form for adding or editing a link definition.
 *
 * This function uses the AHAH Helper module to build a dynamic link definition
 * form. When the AHAH submit function is called and the form is rebuilt this
 * function builds link type specific form elements based on data retrieved with
 * the link type that has been selected.
 */
function autolink_keyword_form_link($form, &$form_state, $plugin, $format, $settings = array()) {
  drupal_add_js(drupal_get_path('module', 'autolink_keyword') . '/autolink_keyword.js', 'file');
  drupal_add_css(drupal_get_path('module', 'autolink_keyword') . '/autolink_keyword.css');

  $form['format'] = array('#type' => 'value', '#value' => $format);
  $form['plugin'] = array('#type' => 'value', '#value' => $plugin);
  $form['#tree'] = TRUE;
  $plugin_info = autolink_keyword_autolink_plugin_info();

  $form['definition'] = array(
    '#type' => 'fieldset',
    '#title' => t('Definition'),
    '#collapsible' => TRUE,
    '#collapsed' => FALSE,
  );
  $form['definition']['keyword'] = array(
    '#type' => 'textfield',
    '#title' => t('Keyword(s)'),
    '#required' => TRUE,
    '#default_value' => isset($settings['keyword']) ? $settings['keyword'] : '',
    '#description' => t('Enter a keyword or several words that should be replaced with a link.'),
  );
  $form['definition']['synonyms'] = array(
    '#type' => 'textfield',
    '#title' => t('Synonyms'),
    '#default_value' => isset($settings['synonyms']) ? implode(', ', $settings['synonyms']) : '',
    '#description' => t('Enter a comma-separated list of keyword synonyms. Synonyms will be treated as matches.'),
  );

  // Display an additional selector if there are multiple forms for the plugin.
  if (count($plugin_info[$plugin]['callbacks']['definition forms']) > 1) {
    // Set the method for edit forms or from the form storage.
    if (!isset($form_state['values']['definition']['method'])) {
      $selected_method = isset($settings['method']) ? $settings['method'] : 'no_method';
    }
    else {
      $selected_method = $form_state['values']['definition']['method'];
    }

    $method_options['no_method'] = t('- Select a method -');
    $options = array();
    foreach ($plugin_info[$plugin]['callbacks']['definition forms'] as $key => $value) {
      $options[$key] = $value['label'];
    }
    $method_options += $options;

    $form['definition']['method'] = array(
      '#type' => 'select',
      '#title' => t('Selection method'),
      '#options' => $method_options,
      '#default_value' => $selected_method,
      '#required' => TRUE,
      '#ajax' => array(
        'event' => 'change',
        'wrapper' => 'method-wrapper',
        'callback' => 'autolink_build_link_form',
        'method' => 'replace',
      ),
    );
    $form['definition']['method_submit'] = array(
      '#type' => 'submit',
      '#value' => t('Select'),
      '#attributes' => array('class' => array('autolink-select-button')),
    );
    $form['definition']['method_wrapper'] = array(
      '#prefix' => '<div id="method-wrapper">',
      '#suffix' => '</div>',
      '#tree' => TRUE,
    );

    if ($selected_method !== 'no_method') {
      $callback = $plugin_info[$plugin]['callbacks']['definition forms'][$selected_method]['callback'];
      $callback($form['definition']['method_wrapper'], $settings);
    }
  }
  else {
    $selected_method = key($plugin_info[$plugin]['callbacks']['definition forms']);
    $form['method'] = array('#type' => 'value', '#value' => $selected_method);
    $callback = $plugin_info[$plugin]['callbacks']['definition forms'][$selected_method]['callback'];
    $callback($form['definition'], $settings);
  }

  $form['submit'] = array('#type' => 'submit', '#value' => t('Save'));
  if (isset($settings['lid'])) {
    $form['delete'] = array('#type' => 'submit', '#value' => t('Delete'));
    $form['lid'] = array('#type' => 'value', '#value' => $settings['lid']);
  }
  return $form;
}

/**
 * AJAX callback for the link add/edit form.
 */
function autolink_build_link_form($form, $form_state) {
  return $form['definition']['method_wrapper'];
}

/**
 * Validation handler for the link edit form.
 *
 * @see autolink_keyword_form_link()
 */
function autolink_keyword_form_link_validate($form, &$form_state) {
  $plugin_info = autolink_keyword_autolink_plugin_info();
  $values = _autolink_merge_link_form_values($form_state['values']);

  $callback = $plugin_info[$values['plugin']]['callbacks']['definition forms'][$values['method']]['callback'];
  $validate = $callback . '_validate';
  if (function_exists($validate)) {
    $validate($values);
  }
}

/**
 * Final submit function for the link form.
 */
function autolink_keyword_form_link_submit($form, &$form_state) {
  $plugin_info = autolink_keyword_autolink_plugin_info();
  $values = _autolink_merge_link_form_values($form_state['values']);

  $callback = $plugin_info[$values['plugin']]['callbacks']['definition forms'][$values['method']]['callback'];
  $submit = $callback . '_submit';
  if (function_exists($submit)) {
    $submit($values);
  }

  switch (autolink_keyword_link_save($values)) {
    case SAVED_NEW:
      drupal_set_message(t('Created new link for keyword <i>%keyword</i>.', array('%keyword' => $values['keyword'])));
      watchdog('autolink', 'Created new link for keyword <i>%keyword</i>.', array('%keyword' => $values['keyword']), WATCHDOG_NOTICE, l(t('edit'), "admin/config/content/{$values['plugin']}/{$values['format']}/edit/{$values['lid']}"));
      break;
    case SAVED_UPDATED:
      drupal_set_message(t('Updated link for keyword <i>%keyword</i>.', array('%keyword' => $values['keyword'])));
      watchdog('autolink', 'Updated link for keyword <i>%keyword</i>.', array('%keyword' => $values['keyword']), WATCHDOG_NOTICE, l(t('edit'), "admin/config/content/{$values['plugin']}/{$values['format']}/edit/{$values['lid']}"));
      break;
  }
  drupal_goto("admin/config/content/formats/{$values['format']}/{$values['plugin']}/settings");
}

/**
 * Form builder for the link delete confirmation form.
 *
 * @param $lid
 *   The ID of a link.
 *
 * @ingroup forms
 * @see autolink_confirm_link_delete_submit()
 */
function autolink_keyword_confirm_link_delete($form, &$form_state, $plugin, $format, $link) {
  $form['format'] = array('#type' => 'value', '#value' => $format);
  $form['plugin'] = array('#type' => 'value', '#value' => $plugin);
  $form['lid'] = array('#type' => 'value', '#value' => $link['lid']);
  $form['keyword'] = array('#type' => 'value', '#value' => $link['keyword']);
  return confirm_form($form,
                  t('Are you sure you want to delete %keyword?',
                  array('%keyword' => $link['keyword'])),
                  "admin/config/content/formats/$format/$plugin/settings",
                  t('This action cannot be undone.'),
                  t('Delete'),
                  t('Cancel'));
}

/**
 * Submit handler to delete a link after confirmation.
 *
 * @see autolink_confirm_link_delete()
 */
function autolink_keyword_confirm_link_delete_submit($form, &$form_state) {
  // Delete the link and record the deletion
  $status = autolink_keyword_link_delete($form_state['values']['lid']);

  drupal_set_message(t('Deleted link definition for keyword <i>%link</i>.', array('%link' => $form_state['values']['keyword'])));
  watchdog('autolink', 'Deleted link definition for keyword <i>%link</i>.', array('%link' => $form_state['values']['keyword']), WATCHDOG_NOTICE);
  $form_state['redirect'] = 'admin/config/content/formats/' . $form_state['values']['format'] . '/' . $form_state['values']['plugin'] . '/settings';

  return;
}

/**
 * Merges AJAX form values into a readable $link array in preparation for database insert.
 */
function _autolink_merge_link_form_values($form_values) {
  $link = $form_values;
  $link += $form_values['definition'];
  if (isset($form_values['definition']['method_wrapper'])) {
    $link += $form_values['definition']['method_wrapper'];
  }
  return $link;
}

/**
 * Defines the external url form.
 */
function autolink_keyword_external_url_form(&$form, $settings) {
  $form['path'] = array(
    '#type' => 'textfield',
    '#title' => t('URL'),
    '#required' => TRUE,
    '#default_value' => isset($settings['path']) ? $settings['path'] : '',
    '#description' => t('e.g. <i>http://www.example.com</i>'),
  );
}

/**
 * Validates the external url form.
 */
function autolink_keyword_external_url_form_validate($form_values) {
  if (!url_is_external($form_values['path'])) {
    form_set_error('path', t('The url <i>%url</i> is invalid. You must provide a valid external url.', array('%url' => $form_values['path'])));
  }
}

/**
 * Defines the node select form.
 */
function autolink_keyword_node_select_form(&$form, $settings) {
  if (isset($settings['path']) && !empty($settings['path'])) {
    $args = explode('/', $settings['path']);
    $nid = $args[1];
  }

  $nodes = array();
  $result = db_query('SELECT n.nid, n.title FROM {node} n');
  while ($node = db_fetch_object($result)) {
    $nodes[$node->nid] = $node->title;
  }
  asort($nodes);

  $select['select'] = t('--Select a node--');
  $select += $nodes;

  $form['title'] = array(
    '#type' => 'select',
    '#title' => t('Select a node title'),
    '#options' => $select,
    '#default_value' => isset($nid) ? $nid : NULL,
  );
}

/**
 * Validates the node select form.
 */
function autolink_keyword_node_select_form_validate($form_values) {
  if ($form_values['title'] == 'select') {
    form_set_error('title', t('You must select content'));
  }
}

/**
 * Submit handler for the node select form.
 */
function autolink_keyword_node_select_form_submit(&$form_values) {
  $form_values['path'] = 'node/' . $form_values['title'];
}

/**
 * Defines the node path form.
 */
function autolink_keyword_node_path_form(&$form, $settings) {
  $form['path'] = array(
    '#type' => 'textfield',
    '#title' => t('Enter a path'),
    '#required' => TRUE,
    '#default_value' => isset($settings['path']) ? $settings['path'] : '',
    '#description' => t('Provide a valid Drupal node path, normal or aliased.<br />' .
                        'Example: \'node/12\' or \'foo/bar\''),
  );
}

/**
 * Validates the node path form.
 */
function autolink_keyword_node_path_form_validate($form_values) {
  if (!drupal_valid_path($form_values['path']) && !drupal_lookup_path('source', $form_values['path']) && $form_values['path'] !== 'contact') {
    form_set_error('path', t('The path <i>%path</i> is invalid. You must provide a valid internal path.', array('%path' => $form_values['path'])));
  }
}

/**
 * Defines the user name form.
 */
function autolink_keyword_user_name_form(&$form, $settings) {
  if (isset($settings['path'])) {
    $args = explode('/', $settings['path']);
    $uid = $args[1];
    $user = is_numeric($uid) ? user_load($uid) : NULL;
  }

  $form['user_name'] = array(
    '#type' => 'textfield',
    '#title' => t('Enter a username'),
    '#required' => TRUE,
    '#default_value' => isset($user) && !is_null($user) ? $user->name : '',
  );
}

/**
 * Validates the user name form.
 */
function autolink_keyword_user_name_form_validate($form_values) {
  $result = db_select('user', 'u')
    ->fields('u')
    ->condition('name', $form_values['user_name'], '=')
    ->execute()->fetchObject();
  if (empty($result)) {
    form_set_error('user_name', t('The user %user does not exist.', array('%user' => $form_values['user_name'])));
  }
}

/**
 * Submit handler for the user name form.
 */
function autolink_keyword_user_name_form_submit(&$form_values) {
  $result = db_select('user', 'u')
    ->fields('u')
    ->condition('name', $form_values['user_name'], '=')
    ->execute();
  if ($user = $result->fetchObject()) {
    $form_values['path'] = 'user/' . $user->uid;
  }
}

/**
 * Defines the user select form.
 */
function autolink_keyword_user_select_form(&$form, $settings) {
  if (isset($settings['path'])) {
    $args = explode('/', $settings['path']);
    $uid = $args[1];
  }

  $options = array('select' => t('- Select a user -'));
  $users = array();
  $result = db_select('users', 'u')
    ->fields('u', array('uid', 'name'))
    ->execute();
  foreach ($result as $user) {
    $users[$user->uid] = $user->name;
  }
  asort($users);
  $options += $users;

  $form['user'] = array(
    '#type' => 'select',
    '#title' => t('Users'),
    '#required' => TRUE,
    '#options' => $options,
    '#default_value' => isset($uid) ? $uid : 'select',
    '#description' => t('Links will link to the user profile of the selected user.'),
  );
}

/**
 * Validates the user select form.
 */
function autolink_keyword_user_select_form_validate($form_values) {
  if ($form_values['user'] == 'select') {
    form_set_error('user', t('You must select a user'));
  }
}

/**
 * Validates the user select form.
 */
function autolink_keyword_user_select_form_submit(&$form_values) {
  $form_values['path'] = 'user/' . $form_values['user'];
}

/**
 * Defines the user mail form.
 */
function autolink_keyword_user_mail_form(&$form, $settings) {
  if (isset($settings['path'])) {
    $args = explode('/', $settings['path']);
    $uid = $args[1];
    $user = is_numeric($uid) ? user_load($uid) : NULL;
  }

  $form['user_mail'] = array(
    '#type' => 'textfield',
    '#title' => t('Enter a user e-mail address'),
    '#required' => TRUE,
    '#default_value' => isset($user) && !is_null($user) ? $user->mail : '',
  );
}

/**
 * Validates the user mail form.
 */
function autolink_keyword_user_mail_form_validate($form_values) {
  $result = db_select('user', 'u')
    ->fields('u')
    ->condition('mail', $form_values['user_mail'], '=')
    ->execute()->fetchObject();
  if (empty($result)) {
    form_set_error('user_mail', t('Cannot locate user data for user email %email.', array('%email' => $form_values['user_mail'])));
  }
}

/**
 * Submit handler for the user mail form.
 */
function autolink_keyword_user_mail_form_submit(&$form_values) {
  $user = db_select('user', 'u')
    ->fields('u')
    ->condition('mail', $form_values['user_mail'], '=')
    ->execute()->fetchObject();
  $form_values['path'] = 'user/' . $user->uid;
}

/**
 * Defines the term select form.
 */
function autolink_keyword_term_select_form(&$form, $settings) {
  if (isset($settings['path'])) {
    $args = explode('/', $settings['path']);
  }

  $select['select'] = t('--Select a term--');

  $terms = array();
  $result = db_select('taxonomy_term_data', 't')
    ->fields('t')->execute();
  foreach ($result as $term) {
    $terms[$term->tid] = $term->name;
  }
  asort($terms);

  $select += $terms;

  $form['term_select'] = array(
    '#type' => 'select',
    '#title' => t('Select a term'),
    '#required' => TRUE,
    '#options' => $select,
    '#default_value' => isset($args[2]) ? $args[2] : 'select',
  );
}

/**
 * Validates the term select form.
 */
function autolink_keyword_term_select_form_validate($form_values) {
  if ($form_values['term_select'] == 'select') {
    form_set_error('term_select', t('You must select a term.'));
  }
}

/**
 * Submit handler for the term select form.
 */
function autolink_keyword_term_select_form_submit(&$form_values) {
  $form_values['path'] = 'taxonomy/term/' . $form_values['term_select'];
}

/**
 * Defines the term multiple form.
 */
function autolink_keyword_term_multiple_form(&$form, $settings) {
    if (!empty($settings['path'])) {
    // Get the term names from tids by parsing the path.
    $args = explode('/', $settings['path']);
    $tids = explode(',', $args[2]);

    foreach ($tids as $tid) {
      $term = taxonomy_term_load($tid);
      $terms[$tid] = $term->name;
    }

    asort($terms);
    $terms_default = implode(', ', $terms);
  }

  $form['terms'] = array(
    '#type' => 'textfield',
    '#title' => t('Taxonomy term(s)'),
    '#required' => TRUE,
    '#default_value' => isset($terms_default) ? $terms_default : '',
    '#description' => t('Enter a comma separated list of taxonomy terms to which the anchor should link.'),
  );
  $form['operator'] = array(
    '#type' => 'radios',
    '#title' => t('Operation'),
    '#options' => array(
      'and' => 'and',
      'or' => 'or',
    ),
    '#default_value' => isset($settings['operator']) ? $settings['operator'] : '',
  );
}

/**
 * Validates the term multiple form.
 */
function autolink_keyword_term_multiple_form_validate($form_values) {
  $terms = drupal_explode_tags($form_values['terms']);

  // Ensure an operation is selected if multiple terms were entered.
  if (isset($terms[1]) && empty($form_values['operator'])) {
    form_set_error('operator', t('You must select an operation for multiple term definitions.'));
  }

  // Check to ensure terms exist.
  $result = db_select('taxonomy_term_data', 't')->fields('t');
  foreach ($terms as $term) {
    if (!$result->condition('name', $term, '=')->execute()->fetchObject()) {
      form_set_error('terms', t('Cannot locate term data for <i>%term</i>. Please enter another term.', array('%term' => $term)));
    }
  }
  unset($term);
}

/**
 * Submit handler for the term multiple form.
 */
function autolink_keyword_term_multiple_form_submit(&$form_values) {
  // Get term IDs for use in urls.
  $terms = drupal_explode_tags($link['terms']);
  foreach ($terms as $name) {
    $result = db_select('taxonomy_term_data', 't')
      ->fields('t')
      ->condition('name', $name, '=')
      ->execute();
    if ($term = $result->fetchObject()) {
      $tids[] = $term->tid;
    }
  }

  // If this is only one term then we don't need an operation.
  if (!isset($terms[1])) {
    $arg = $tids[0];
  }
  else {
    // This is a multiple term url, so it needs to be built with operators (,) and (+).
    switch ($form_values['operator']) {
      case 'and':
        $arg = implode(',', $tids);
        break;
      case 'or':
        $arg = implode('+', $tids);
        break;
    }
  }

  $form_values['path'] = 'taxonomy/term/' . $arg;
}

/**
 * Defines the node search form.
 */
function autolink_keyword_node_search_form(&$form, $settings) {
  $form['search_terms'] = array(
    '#type' => 'textfield',
    '#title' => t('Search terms'),
    '#required' => TRUE,
    '#default_value' => isset($settings['query']) ? $settings['query'] : '',
    '#description' => t('The search terms whose results users should be directed upon clicking the link.'),
  );
}

/**
 * Submits the node search link form.
 */
function autolink_keyword_node_search_form_submit(&$form_values) {
  $form_values['path'] = 'search/node';
  $form_values['query'] = $form_values['search_terms'];
}

/**
 * Defines the user search form.
 */
function autolink_keyword_user_search_form(&$form, $settings) {
  $form['search_terms'] = array(
    '#type' => 'textfield',
    '#title' => t('Search terms'),
    '#required' => TRUE,
    '#default_value' => isset($settings['query']) ? $settings['query'] : '',
    '#description' => t('The search terms whose results users should be directed upon clicking the link.'),
  );
}

/**
 * Submits the user search link form.
 */
function autolink_keyword_user_search_form_submit(&$form_values) {
  $form_values['path'] = 'search/user';
  $form_values['query'] = $form_values['search_terms'];
}
