<?php

namespace Drupal\automated_testing_kit_demo\EventSubscriber;

use Drupal\Core\Recipe\RecipeAppliedEvent;
use Drupal\Core\CronInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Class RecipeEventSubscriber.
 */
class RecipeEventSubscriber implements EventSubscriberInterface {

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The cron service.
   *
   * @var \Drupal\Core\CronInterface
   */
  protected $cron;

   /**
   * The messenger service.
   *
   * @var \Drupal\Core\CronInterface
   */
  protected $messenger;

  /**
   * Constructs a new RecipeEventSubscriber object.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   */
  public function __construct(LoggerChannelFactoryInterface $logger_factory, CronInterface $cron, MessengerInterface $messenger) {
    $this->loggerFactory = $logger_factory;
    $this->cron = $cron;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      RecipeAppliedEvent::class => ['onRecipeApplied'],
    ];
  }

  /**
   * React to the ATK Drupal Demo recipe being applied.
   *
   * @param \Drupal\automated_testing_kit_demo\Event\RecipeAppliedEvent $event
   *   The event object.
   */
  public function onRecipeApplied(RecipeAppliedEvent $event) {
    $recipe = $event->recipe;
    $name = $recipe->name;

    if ($name == 'Automated Testing Kit - Demonstration Recipe') {
      $this->loggerFactory->get('automated_testing_kit_demo')->info('Recipe @name was applied', ['@name' => $name]);

      // Run cron to index the site.
      $this->cron->run();

      // Adjust the Drush alias to work with DDEV.
      $filepath = '../playwright.atk.config.js';

      if (file_exists($filepath) && is_writable($filepath)) {
        $lines = file($filepath);
        if ($lines !== false) {
            // Search for the line containing drushCmd
            $drushCmdLine = -1;
            foreach ($lines as $i => $line) {
                if (strpos($line, 'drushCmd') !== false) {
                    $drushCmdLine = $i;
                    break;
                }
            }

            if ($drushCmdLine !== -1) {
                $lines[$drushCmdLine] = "  drushCmd: \"ddev drush\",\n";

                if (file_put_contents($filepath, implode('', $lines))) {
                  $this->messenger->addMessage('File updated successfully.');
                } else {
                  $this->messenger->addError('Failed to write to file.');
                }
            } else {
              $this->messenger->addError('Could not find drushCmd line in the file.');
            }
        } else {
          $this->messenger->addError('Failed to read file.');
        }
      } else {
        $this->messenger->addError('File does not exist or is not writable.');
      }
    }
  }
}