/**
 * atk_register_login.spec.js
 *
 * Registration, login and forgotten password tests.
 */

/** ESLint directives */
/* eslint-disable import/first */

import * as atkCommands from '../support/atk_commands'
import * as atkUtilities from '../support/atk_utilities'

// Set up Playwright.
const { test } = require('@playwright/test')

import playwrightConfig from '../../playwright.config'

const baseUrl = playwrightConfig.use.baseURL

// Import ATK Configuration.
import atkConfig from '../../playwright.atk.config'

// Standard accounts that use user accounts created
// by QA Accounts. QA Accounts are created when the QA
// Accounts module is enabled.
import qaUserAccounts from '../data/qaUsers.json'

test.describe('User registration and login tasks.', () => {
  // Temporary users that should be cleaned up after the test run.
  const tmpUsers = []

  //
  // Register the created user.
  //
  test('(ATK-PW-1000) Register with form. @ATK-PW-1000 @register-login @alters-db @smoke', async ({ page }) => {
    const testId = 'ATK-PW-1000'

    // Make up a new user object.
    const randomUser = atkUtilities.createRandomUser()
    tmpUsers.push(randomUser)
    const { userName } = randomUser
    const { userEmail } = randomUser

    // Begin registration.
    await page.goto(baseUrl + atkConfig.registerUrl)

    await page.getByLabel('Email address').fill(userEmail)
    await page.getByLabel('Username').fill(userName)
    await page.getByRole('button', { name: 'Create new account' }).click()

    // The status box needs a moment to appear.
    await atkCommands.expectMessage(page, 'Thank you for applying for an account.')
  })

  //
  // Log in with the login form into the authenticated account.
  //
  test('(ATK-PW-1010) Log in via login form. @ATK-PW-1010 @register-login @smoke', async ({ page, context }) => {
    const testId = 'ATK-PW-1010'

    await atkCommands.logInViaForm(page, context, qaUserAccounts.authenticated)
  })

  //
  // Log in with a POST request into the authenticated account.
  //
  test('(ATK-PW-1011) Log in via POST. @ATK-PW-1011 @register-login @smoke', async ({ page }) => {
    const testId = 'ATK-PW-1011'

    // TODO: Not ready yet.
    // await atkCommands.logInViaPost(page, context, request, qaUserAccounts.authenticated)
  })

  //
  // Log in with a ULI generated by Drupal.
  //
  test('(ATK-PW-1012) Log in via ULI. @ATK-PY-1012 @register-login @smoke', async ({ page, context }) => {
    const testId = 'ATK-PW-1012'

    await atkCommands.logInViaUli(page, context, 1)
  })

  //
  // Validate reset password function.
  //
  test('(ATK-PW-1030) Reset password. @ATK-PW-1030 @register-login @smoke', async ({ page }) => {
    const testId = 'ATK-PW-1030'

    const resetAccount = atkUtilities.createRandomUser()
    tmpUsers.push(resetAccount)
    await atkCommands.createUserWithUserObject(resetAccount, [])

    // Begin registration.
    await page.goto(baseUrl + atkConfig.resetPasswordUrl)

    await page.getByLabel('Username').fill(resetAccount.userName)
    await page.getByRole('button', { name: 'Submit' }).click()

    // The status box needs a moment to appear.
    await atkCommands.expectMessage(page, 'an email will be sent with instructions to reset your password.')
  })

  test.afterAll(() => {
    // Uncomment for debugging.
    // console.log(`Cleaning up users:\n${JSON.stringify(tmpUsers, null, 2)}`)
    for (const user of tmpUsers) {
      atkCommands.deleteUserWithUserName(user.userName, ['--delete-content'])
    }
  })
})
