/**
 * atk_sitemap.cy.js
 *
 * Validate sitemap.xml.
 *
 */

/** ESLint directives */
/* eslint-disable import/first */

/// <reference types='Cypress' />

// Set up ATK.
import { XMLParser } from 'fast-xml-parser';
import * as atkCommands from '../../support/atk_commands';
import * as atkUtilities from '../../support/atk_utilities'; // eslint-disable-line no-unused-vars
import atkConfig from '../../../cypress.atk.config'; // eslint-disable-line no-unused-vars

// Standard accounts that use user accounts created
// by QA Accounts. QA Accounts are created when the QA
// Accounts module is enabled.
import qaUserAccounts from '../../data/qaUsers.json';

describe('Sitemap tests.', () => {
  //
  // Confirm at least one sitemap exists; report total number of sitemaps.
  // TODO: Expand by using https://glebbahmutov.com/blog/test-sitemap/.
  //
  it('(ATK-CY-1070) Return # of sitemap files; fail if zero.', { tags: ['@ATK-CY-1070', '@xml-sitemap', '@smoke'] }, () => {
    const testId = 'ATK-CY-1070'; // eslint-disable-line no-unused-vars
    const fileName = 'sitemap.xml';

    // If there isn't at least one sitemap, this will fail.
    cy.request(fileName).its('body').then((body) => {
      const parser = new XMLParser();
      const jsonObj = parser.parse(body);

      let sitemapCount = 1;
      try {
        // If there is just one sitemap file, this will fail.
        sitemapCount = jsonObj.sitemapindex.sitemap.length;
      } catch (error) {}

      console.log(`Total sitemap files: ${sitemapCount}`); // eslint-disable-line no-console
    });
  });

  //
  // Regenerate sitemap files.
  // 1. Find Site ID of default sitemap (change for your installation).
  // 2. Fetch the 1.xml timestamp.
  // 3. Use drush xmlsitemap:regenerate to create new files.
  // 4. Validate new files.
  //
  it('(ATK-CY-1071) Regenerate sitemap files.', { tags: ['@ATK-CY-1071', '@xml-sitemap', '@smoke'] }, () => {
    const testId = 'ATK-CY-1071'; // eslint-disable-line no-unused-vars
    let siteId = null;
    let filename = null;
    let firstFileProps;

    //
    // Step 1.
    //
    cy.logInViaForm(qaUserAccounts.admin);
    cy.visit(atkConfig.xmlSitemapUrl);
    // Remove everything after the domain name.
    cy.url().then((baseUrl) => {
      let match = baseUrl
          .match(/^(?:https?:\/\/)(?:[^@\/\n]+@)?(?:[^:\/\n]+)/);
      if (!match) {
        throw new Error(`Url is not parsed: ${baseUrl}`);
      }
      return match[0];
    }).then((trimmedBaseUrl) => {
      cy.log(`**Looking for XML site map for site name: ${trimmedBaseUrl}**`);
      cy.get(`table tr`).each((tr) => {
        // Examine each row.
        // Check if the first column contains site name.
        if (tr.find('td:nth-child(1)').text().includes(trimmedBaseUrl)) {
          // Get the text content of the second column in that row
          siteId = tr.find('td:nth-child(2)').text();
          return false; // This stops the .each() loop.
        }
      });

    }).then(() => {
      //
      // Step 2.
      //
      filename = `sites/default/files/xmlsitemap/${siteId}/1.xml`; // eslint-disable-line no-unused-vars
      cy.log(`**Evaluating sitemap file: ${filename}**`);
    }).then(() => {
      cy.execDrush(`fprop --format=json ${filename}`);
    }).then((text) => {
      // Capture the timestamp to ensure it changes.
      firstFileProps = JSON.parse(text);
    }).then(() => {
      //
      // Step 3.
      //
      cy.execDrush('xmlsitemap:rebuild');
    }).then(() => {
      cy.execDrush(`fprop --format=json ${filename}`);
    }).then((text) => {
      //
      // Step 4.
      //
      const secondFileProps = JSON.parse(text);
      const firstTime = firstFileProps[0].filemtime;
      const secondTime = secondFileProps[0].filemtime;
      expect(firstTime).not.to.be.undefined;
      expect(secondTime).not.to.be.undefined;
      expect(firstTime).not.to.eq(secondTime);
    });
  });

  //
  // Regenerate sitemap files for SimpleSiteMap.
  // 1080 to 1089 reserved for Simple XML Sitemap (https://www.drupal.org/project/simple_sitemap) tests.
  //
});
