/**
 * atk_register_login.spec.js
 *
 * Registration, login and forgotten password tests.
 */

/** ESLint directives */
/* eslint-disable import/first */

// Set up Playwright.
const { test, expect } = require('@playwright/test');
import playwrightConfig from '../../playwright.config';
const baseUrl = playwrightConfig.use.baseURL;

// Import ATK Libraries and configuration.
import * as atkCommands from '../support/atk_commands';
import * as atkUtilities from '../support/atk_utilities';
import atkConfig from '../../playwright.atk.config';

// Import email settings for Ethereal fake SMTP service.
import userEtherealAccount from '../data/etherealUser.json';

// Standard accounts that use user accounts created
// by QA Accounts. QA Accounts are created when the QA
// Accounts module is enabled.
import qaUserAccounts from '../data/qaUsers.json';

test.describe('User registration and login tasks.', () => {
  //
  // Register the Ethereal user and confirm email reaches Ethereal.
  //
  test('(ATK-PW-1000) Register with form and confirm email with Ethereal. @ATK-PW-1000 @register-login @alters-db @smoke', async ({ page }) => {
    let textContent = null;

    // Clean up user in case it exists. Use email because we modify the name
    // to make it easier to find the registration email in Ethereal.email.
    atkCommands.deleteUserWithEmail(userEtherealAccount.userEmail, ['--delete-content']);

    let a = baseUrl + atkConfig.registerUrl;
    // Begin registration.
    await page.goto(baseUrl + atkConfig.registerUrl);

    await page.getByLabel('Email address').fill(userEtherealAccount.userEmail);
    const extendedUserName = `${userEtherealAccount.userName}-${atkUtilities.createRandomString(6)}`;
    await page.getByLabel('Username').fill(extendedUserName);
    await page.getByRole('button', { name: 'Create new account' }).click();

    // Should see the thank-you message.
    await atkCommands.expectMessage(page, 'Thank you for applying for an account.');

    // Give the email some time to arrive, adjust as needed.
    await page.waitForTimeout(1000);

    // Check for registration email at Ethereal.
    await atkCommands.checkEmail(page, userEmail, `Account details for ${userName}`);

    // Clean up
    atkCommands.deleteUserWithUserName(userName, ['--delete-content']);
  });

  //
  // Log in with the login form into the authenticated account.
  //
  test('(ATK-PW-1010) Log in via login form. @ATK-PW-1010 @register-login @smoke', async ({ page, context }) => {
    await atkCommands.logInViaForm(page, context, qaUserAccounts.authenticated);
  });

  //
  // Log in with a POST request into the authenticated account.
  //
  test('(ATK-PW-1011) Log in via POST. @ATK-PW-1011 @register-login @smoke', async ({ page }) => { // eslint-disable-line no-unused-vars
    // TODO: Not ready yet.
    // await atkCommands.logInViaPost(page, context, request, qaUserAccounts.authenticated)
  });

  //
  // Log in with a ULI generated by Drupal.
  //
  test('(ATK-PW-1012) Log in via ULI. @ATK-PY-1012 @register-login @smoke', async ({ page, context }) => {
    await atkCommands.logInViaUli(page, 1);
  });

  //
  // Validate reset password function.
  //
  test('(ATK-PW-1030) Reset password. @ATK-PW-1030 @register-login @smoke', async ({ page }) => {
    await atkCommands.deleteUserWithEmail(userEtherealAccount.userEmail, ['--delete-content']);

    // Use random string to identify user in Ethereal.email.
    const extendedUserName = `${userEtherealAccount.userName}-${atkUtilities.createRandomString(6)}`;

    const resetAccount = {
      userName: extendedUserName,
      userEmail: userEtherealAccount.userEmail,
      userPassword: userEtherealAccount.userPassword,
      userRoles: [],
    };
    await atkCommands.createUserWithUserObject(resetAccount, []);

    // Begin registration.
    await page.goto(baseUrl + atkConfig.resetPasswordUrl);

    await page.getByLabel('Username').fill(extendedUserName);
    await page.getByRole('button', { name: 'E-mail new password' }).click();

    // The status box needs a moment to appear.
    await atkCommands.expectMessage(page, 'an email will be sent with instructions to reset your password.');

    // Give the email some time to arrive.
    await page.waitForTimeout(1000);

    // Check for reset password email at Ethereal.
    await atkCommands.checkEmail(page, resetAccount.userEmail, `Replacement login information for ${resetAccount.userName}`)

    atkCommands.deleteUserWithUserName(resetAccount.userName, ['--delete-content']);
  });
});
