<?php

namespace Drupal\automatic_taxonomy_terms\Config;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\RouteMatchInterface;

class VocabularyConfig {
  const automatic_taxonomy_terms_CONFIG_PREFIX = 'automatic_taxonomy_terms.';
  /** @var \Drupal\Core\Routing\RouteMatchInterface */
  private $routeMatch;
  /** @var \Drupal\Core\Config\ConfigFactoryInterface */
  private $configFactory;
  /** @var string */
  private $vocabularyConfigName;
  /** @var \Drupal\Core\Config\ImmutableConfig */
  private $vocabularyConfig;
  /** @var \Drupal\Core\Entity\EntityTypeManagerInterface */
  private $entityTypeManager;
  /** @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface */
  private $entityTypeBundleInfo;

  /**
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityTypeBundleInfo
   */
  public function __construct(ConfigFactoryInterface $configFactory, RouteMatchInterface $routeMatch, EntityTypeManagerInterface $entityTypeManager, EntityTypeBundleInfoInterface $entityTypeBundleInfo) {
    $this->routeMatch = $routeMatch;
    $this->configFactory = $configFactory;
    $this->vocabularyConfigName = self::automatic_taxonomy_terms_CONFIG_PREFIX . $this->getTaxonomyVocabularyFromCurrentRoute();
    $this->vocabularyConfig = $configFactory->get($this->vocabularyConfigName);
    $this->entityTypeManager = $entityTypeManager;
    $this->entityTypeBundleInfo = $entityTypeBundleInfo;
  }

  /**
   * @return string
   */
  public function getTaxonomyVocabularyFromCurrentRoute() {
    return $this->routeMatch->getParameter('taxonomy_vocabulary');
  }

  /**
   * @param string $entityTypeId
   * @param string $entityTypeBundleId
   * @return mixed
   */
  public function getBundleConfiguredParentItem(string $entityTypeId, string $entityTypeBundleId) {
    $configuredBundles = $this->vocabularyConfig->get('bundles');

    if ($configuredBundles) {
      $taxonomyTermId = $configuredBundles["{$entityTypeId}:{$entityTypeBundleId}"]['parent'];
      return $this->entityTypeManager->getStorage('taxonomy_term')
        ->load($taxonomyTermId);
    }
  }

  /**
   * @param string $entityTypeId
   * @param string $entityTypeBundleId
   * @return string
   */
  public function getBundleConfiguredTermPattern(string $entityTypeId, string $entityTypeBundleId) {
    return $this->getBundleConfiguredValue($entityTypeId, $entityTypeBundleId, 'label');
  }

  /**
   * @param string $entityTypeId
   * @param string $entityTypeBundleId
   * @param string $option
   * @return mixed
   */
  private function getBundleConfiguredValue(string $entityTypeId, string $entityTypeBundleId, string $option) {
    $configuredBundles = $this->vocabularyConfig->get('bundles');

    if ($configuredBundles && isset($configuredBundles["{$entityTypeId}:{$entityTypeBundleId}"][$option])) {
      return $configuredBundles["{$entityTypeId}:{$entityTypeBundleId}"][$option];
    }
  }

  /**
   * @param $entityTypeId
   * @param $entityTypeBundleId
   * @return string
   */
  public function getBundleConfiguredTermSync(string $entityTypeId, string $entityTypeBundleId) {
    return $this->getBundleConfiguredValue($entityTypeId, $entityTypeBundleId, 'sync');
  }

  /**
   * @param string[] $taxonomyVocabularies
   * @return array
   */
  public function getEditableConfigNames(array $taxonomyVocabularies) {
    $configNames = [];
    foreach (array_keys($taxonomyVocabularies) as $vocabularyId) {
      $configNames[] = self::automatic_taxonomy_terms_CONFIG_PREFIX . $vocabularyId;
    }
    return $configNames;
  }

  /**
   * @return string[]
   */
  public function getEntityTypes() {
    $configuredEntityTypes = $this->vocabularyConfig->get('entity_types');
    return $configuredEntityTypes ? array_filter($configuredEntityTypes) : [];
  }

  /**
   * @return \Drupal\Core\Config\Config
   */
  public function getStorage() {
    return $this->configFactory->getEditable($this->vocabularyConfigName);
  }

  /**
   * @return \Drupal\Core\Config\ImmutableConfig[]
   */
  public function getAllConfigurations() {
    $vocabularyConfigurationNames = [];
    foreach ($this->getTaxonomyVocabularyIds() as $vocabularyName) {
      $vocabularyConfigurationNames[$vocabularyName] = $this->configFactory->get(self::automatic_taxonomy_terms_CONFIG_PREFIX . $vocabularyName);
    }
    return $vocabularyConfigurationNames;
  }

  /**
   * @return string[]
   */
  private function getTaxonomyVocabularyIds() {
    return array_keys($this->entityTypeBundleInfo->getBundleInfo('taxonomy_term'));
  }

}