<?php

namespace Drupal\automatic_taxonomy_terms\Event;

use Drupal\automatic_taxonomy_terms\Config\EntityBundleConfiguration;
use Drupal\automatic_taxonomy_terms\Config\VocabularyConfig;
use Drupal\automatic_taxonomy_terms\Storage\TaxonomyTermSyncStorage;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\hook_event_dispatcher\Event\Entity\EntityDeleteEvent;
use Drupal\hook_event_dispatcher\Event\Entity\EntityInsertEvent;
use Drupal\hook_event_dispatcher\Event\Entity\EntityUpdateEvent;
use Drupal\hook_event_dispatcher\HookEventDispatcherEvents;
use Drupal\token\Token;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class EventSubscriber implements EventSubscriberInterface {
  /** @var \Drupal\automatic_taxonomy_terms\Config\VocabularyConfig */
  private $vocabularyConfig;
  /** @var \Drupal\token\TokenInterface */
  private $token;
  /** @var \Drupal\Core\Entity\EntityTypeManagerInterface */
  private $entityTypeManager;
  /** @var \Drupal\Core\Entity\EntityFieldManagerInterface */
  private $entityFieldManager;

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      HookEventDispatcherEvents::ENTITY_INSERT => ['onEntityCreate', 20],
      HookEventDispatcherEvents::ENTITY_UPDATE => ['onEntityUpdate', 20],
      HookEventDispatcherEvents::ENTITY_DELETE => ['onEntityDelete', 20]
    ];
  }

  /**
   * @param \Drupal\automatic_taxonomy_terms\Config\VocabularyConfig $vocabularyConfig
   * @param \Drupal\token\Token $token
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   */
  public function __construct(VocabularyConfig $vocabularyConfig, Token $token, EntityTypeManagerInterface $entityTypeManager, EntityFieldManagerInterface $entityFieldManager) {
    $this->vocabularyConfig = $vocabularyConfig;
    $this->token = $token;
    $this->entityTypeManager = $entityTypeManager;
    $this->entityFieldManager = $entityFieldManager;
  }

  /**
   * @param \Drupal\hook_event_dispatcher\Event\Entity\EntityInsertEvent $entityEvent
   */
  public function onEntityCreate(EntityInsertEvent $entityEvent) {
    $entity = $entityEvent->getEntity();

    foreach ($this->getBundleConfigurationsForEntity($entity) as $bundleConfiguration) {
      $taxonomyTermSyncEntity = new TaxonomyTermSyncStorage($entity, $bundleConfiguration, $this->entityTypeManager, $this->entityFieldManager);
      $taxonomyTermSyncEntity->create();
    }
  }

  /**
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @return EntityBundleConfiguration[]
   */
  private function getBundleConfigurationsForEntity(EntityInterface $entity) {
    $bundleConfigurations = [];
    foreach ($this->vocabularyConfig->getAllConfigurations() as $vocabularyName => $configuration) {
      $entityBundleConfigurationValues = $configuration->get("bundles.{$entity->getEntityTypeId()}:{$entity->bundle()}");
      if (is_array($entityBundleConfigurationValues)) {
        $bundleConfigurations[] = new EntityBundleConfiguration($configuration, $this->token, $entity, $entityBundleConfigurationValues + ['vocabulary' => $vocabularyName]);
      }
    }

    return $bundleConfigurations;
  }

  /**
   * @param \Drupal\hook_event_dispatcher\Event\Entity\EntityUpdateEvent $entityEvent
   */
  public function onEntityUpdate(EntityUpdateEvent $entityEvent) {
    $entity = $entityEvent->getEntity();

    foreach ($this->getBundleConfigurationsForEntity($entity) as $bundleConfiguration) {
      if ($bundleConfiguration->keepInSync()) {
        $taxonomyTermSyncEntity = new TaxonomyTermSyncStorage($entity, $bundleConfiguration, $this->entityTypeManager, $this->entityFieldManager);
        $taxonomyTermSyncEntity->update();
      }
    }
  }

  /**
   * @param \Drupal\hook_event_dispatcher\Event\Entity\EntityDeleteEvent $entityEvent
   */
  public function onEntityDelete(EntityDeleteEvent $entityEvent) {
    $entity = $entityEvent->getEntity();

    foreach ($this->getBundleConfigurationsForEntity($entity) as $bundleConfiguration) {
      if ($bundleConfiguration->keepInSync()) {
        $taxonomyTermSyncEntity = new TaxonomyTermSyncStorage($entity, $bundleConfiguration, $this->entityTypeManager, $this->entityFieldManager);
        $taxonomyTermSyncEntity->delete();
      }
    }
  }
}