<?php

namespace Drupal\automatic_taxonomy_terms\Storage;

use Drupal\automatic_taxonomy_terms\Config\EntityBundleConfiguration;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;

class TaxonomyTermSyncStorage {
  use StringTranslationTrait;

  const ENTITY_SYNC_FIELD = 'automatic_entity_creator';
  /** @var \Drupal\Core\Entity\EntityInterface */
  private $entity;
  /** @var \Drupal\automatic_taxonomy_terms\Config\EntityBundleConfiguration */
  private $bundleConfiguration;
  /** @var \Drupal\Core\Entity\EntityTypeManagerInterface */
  private $entityTypeManager;
  /** @var \Drupal\taxonomy\TermInterface|null */
  private $syncedTaxonomyTerm = NULL;
  /** @var \Drupal\taxonomy\TermStorageInterface */
  private $termStorage;
  /** @var \Drupal\Core\Entity\EntityFieldManagerInterface */
  private $entityFieldManager;

  /**
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param \Drupal\automatic_taxonomy_terms\Config\EntityBundleConfiguration $bundleConfiguration
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   */
  public function __construct(EntityInterface $entity, EntityBundleConfiguration $bundleConfiguration, EntityTypeManagerInterface $entityTypeManager, EntityFieldManagerInterface $entityFieldManager) {
    $this->entity = $entity;
    $this->bundleConfiguration = $bundleConfiguration;
    $this->entityTypeManager = $entityTypeManager;
    $this->syncedTaxonomyTerm = $this->loadSyncedTaxonomyTerm();
    $this->termStorage = $entityTypeManager->getStorage('taxonomy_term');
    $this->entityFieldManager = $entityFieldManager;
  }

  /**
   * @return \Drupal\Core\Entity\EntityInterface|null
   */
  private function loadSyncedTaxonomyTerm() {
    $taxonomyTerms = $this->entityTypeManager->getStorage('taxonomy_term')
      ->loadByProperties([
        'vid' => $this->bundleConfiguration->getVocabularyName(),
        self::ENTITY_SYNC_FIELD => $this->entity->uuid()
      ]);

    return is_array($taxonomyTerms) ? reset($taxonomyTerms) : NULL;
  }

  public function create() {
    $this->ensureTaxonomyTermSyncField();

    $taxonomyTerm = $this->termStorage->create([
      'name' => $this->bundleConfiguration->label(),
      'vid' => $this->bundleConfiguration->getVocabularyName(),
      'parent' => $this->bundleConfiguration->getTaxonomyTermParentId(),
      'langcode' => $this->entity->language()->getId(),
      self::ENTITY_SYNC_FIELD => $this->entity->uuid()
    ]);
    $taxonomyTerm->save();
  }

  private function ensureTaxonomyTermSyncField() {
    if (!$this->taxonomyFieldStorageExists(self::ENTITY_SYNC_FIELD)) {
      FieldStorageConfig::create(array(
        'field_name' => self::ENTITY_SYNC_FIELD,
        'entity_type' => 'taxonomy_term',
        'type' => 'text',
      ))->save();
    }

    if (!$this->taxonomyFieldExists(self::ENTITY_SYNC_FIELD)) {
      FieldConfig::create(array(
        'field_name' => self::ENTITY_SYNC_FIELD,
        'entity_type' => 'taxonomy_term',
        'bundle' => $this->bundleConfiguration->getVocabularyName(),
        'label' => $this->t('Automatic taxonomy term creation entity')
      ))->save();
    }
  }

  /**
   * @param string $fieldName
   * @return bool
   */
  private function taxonomyFieldStorageExists(string $fieldName) {
    return (bool) $this->entityTypeManager->getStorage('field_storage_config')
      ->load("taxonomy_term.{$fieldName}");
  }

  /**
   * @param string $fieldName
   * @return bool
   */
  private function taxonomyFieldExists(string $fieldName) {
    $fieldDefinitions = $this->entityFieldManager->getFieldDefinitions('taxonomy_term', $this->bundleConfiguration->getVocabularyName());
    return array_key_exists($fieldName, $fieldDefinitions);
  }

  public function update() {
    if ($this->syncedTaxonomyTerm) {
      $this->syncedTaxonomyTerm->set('name', $this->bundleConfiguration->label());
      $this->syncedTaxonomyTerm->save();
    }
  }

  public function delete() {
    if ($this->syncedTaxonomyTerm) {
      $this->syncedTaxonomyTerm->delete();
    }
  }
}