<?php

/**
 * Missing project info checker.
 */
class MissingProjectInfo implements ReadinessCheckerInterface {

  /**
   * {@inheritdoc}
   */
  public static function run() {
    return static::missingProjectInfoCheck();
  }

  /**
   * Check for projects missing project info.
   *
   * @return array
   *   An array of translatable strings if any checks fail.
   */
  protected static function missingProjectInfoCheck() {
    $messages = [];
    foreach (static::getInfos() as $extension_name => $info) {
      if (static::isIgnoredPath($info['extension_uri'])) {
        continue;
      }
      if (!static::getExtensionVersion($extension_name, $info)) {
        $messages[] = t('The project "@extension" can not be updated because its version is either undefined or a dev release.', ['@extension' => $extension_name]);
      }
    }
    return $messages;
  }

  /**
   * Get the extension types.
   *
   * @return array
   *   The extension types.
   */
  protected static function getExtensionsTypes() {
    return ['modules', 'profiles', 'themes'];
  }

  /**
   * Returns an array of info files information of available extensions.
   *
   * @return array
   *   An associative array of extension information arrays, keyed by extension
   *   name.
   */
  protected static function getInfos() {
    $infos = [];
    // Find extensions.
    $extensions = drupal_system_listing('/^' . DRUPAL_PHP_FUNCTION_PATTERN . '\.info$/', 'modules', $key = 'name', $min_depth = 1);
    $extensions = array_merge($extensions, drupal_system_listing('/^' . DRUPAL_PHP_FUNCTION_PATTERN . '\.info$/', 'themes', $key = 'name', $min_depth = 1));
    foreach ($extensions as $extension) {
      if (file_exists($info_file = dirname($extension->uri) . '/' . $extension->name . '.info')) {
        // Get the .info file for the module or theme this file belongs to.
        $infos[$extension->name] = drupal_parse_info_file($info_file);
        $infos[$extension->name]['extension_uri'] = $extension->uri;
      }
    }

    return $infos;
  }

  /**
   * Get the extension version.
   *
   * @param string $extension_name
   *   The extension name.
   * @param array $info
   *   The extension's info.
   *
   * @return string|null
   *   The version or NULL if undefined.
   */
  protected static function getExtensionVersion($extension_name, array $info) {
    if (isset($info['version']) && strpos($info['version'], '-dev') === FALSE) {
      return $info['version'];
    }
  }

  /**
   * Check if the file path is ignored.
   *
   * @param string $file_path
   *   The file path.
   *
   * @return bool
   *   TRUE if file path is ignored, else FALSE.
   */
  protected static function isIgnoredPath($file_path) {
    $paths = variable_get('automatic_updates_ignored_paths', "sites/all/modules/custom/*\nsites/all/themes/custom/*");
    if (drupal_match_path($file_path, $paths)) {
      return TRUE;
    }
  }
}
