(function($) {
  /**
   * Helper method that extends the String object with a padLeft method
   */
  String.prototype.padLeft = function(value, size) {
    var x = this;
    while (x.length < size) {
      x = value + x;
    }
    return x;
  };

  /**
   * Helper method that extends the Date object with a toFormattedString method to allow for
   * easier printing of dates. (Reduced version of one that was found on Stackoverflow.)
   */
  Date.prototype.toFormattedString = function(format) {
    return format
      .replace(/yyyy/g, this.getFullYear())
      .replace(/mm/g, String(this.getMonth() + 1).padLeft('0', 2))
      .replace(/m/g, String(this.getMonth() + 1))
      .replace(/dd/g, String(this.getDate()).padLeft('0', 2))
      .replace(/d/g, this.getDate());
  };

  /**
   * Javascript API for availability Calendars module.
   *
   * @class AvailabilityCalendars
   *   Represents the client-side interface to an availability calendar
   *   for a given node. In principal, it is possible to have multiple calendars for differents
   *   nodes on the same page. If this is the case, use the nid parameter in the constructor to
   *   specify which calendar.
   *
   *   This API object provides a way of client-side interacting with the calendar. Basically it
   *   provides:
   *   - Some methods to change the (visual) status of calendar days. Note however, that it does not
   *     update the server-side calendar.
   *   - It defines a 'calendarclick' event and triggers it when the visitor clicks on a day cell in
   *     the calendar. The 'calendarclick' event passes in a Date object, representing the day that
   *     was clicked on, and the nid, to isdentify which calendar was clicked on. However, as only
   *     DOM elements can trigger events, you must bind your custom event handler to the calendar
   *     element, retrieved via AvailabilityCalendar.getCalendar():
   *     @code
   *     myAvailabilityCalendar.getCalendar().bind('calendarclick', function(event, date, nid) {
   *       alert('You clicked on date ' + date + ' of node ' + nid);
   *     });
   *     @endcode
   *
   * @constructor
   *   Creates a new AvailibilityCalendars interaction object.
   * @param object
   *   Settings Required: object with the following properties:
   *   {
   *     splitDay, (boolean: indicating whether this calendar allows split days)
   *     states {
   *       class* { (1 property for each available availability state)
   *         class,
   *         label,
   *         weight
   *       }
   *     }
   *   }
   * @param int nid
   *   Optional: the node id for the Calendar we want to interact with. If not
   *   provided, the first calendar on the page is taken.
   */
  var AvailabilityCalendars = (function(settings, nid) {
    var _settings = settings;
    var _calendarId = '#' + $(nid !== undefined ? '#availability-calendar-' + nid : '.availability-calendar:first').attr('id');
    var _nid = _calendarId.split('-', 3)[2];
    var _calendarRange = null;
    _initCustomEvents();

    /**
     * Initializes the custom events for this calendar.
     * The events are triggered by the calendar element, the div surrounding this calendar.
     * Other javascript thus should bind to that element (retrieved via @see getCalendar()).
     *
     * Currently provided custom events:
     * - calendarclick: comes with a date object and the nid as parameters.
     */
    function _initCustomEvents() {
      getCalendar().click(function(event) {
        // Find out if event originated from a day cell, get the date,
        // and trigger the event on the calendar element.
        var day, month, year;
        $(event.target).closest('td')
          .filter(function() {
            cell = $(this);
            return !cell.hasClass('calweeknote') && cell.html().search(/^<span>[1-3]?[0-9]</i) === 0;
          })
          .each(function() {
            day = Number($(this).text());
          })
          .closest('table.calmonth')
          .filter(function() {
            return $(this).attr('id').split('-', 4).length == 4;
          })
          .each(function() {
            var idParts = $(this).attr('id').split('-', 4);
            year = idParts[1];
            month = idParts[2];
          })
          .closest(_calendarId)
          .triggerHandler('calendarclick', [new Date(year, month - 1, day), _nid]);
      });
    };

    /**
     * @returns jQuery
     *   A jQuery object containing the calendar DOM element, that is the div
     *   with id='availability-calendar-{nid}' and class='availability-calendar' surrounding
     *   this calendar. This element triggers the custom events, thus other javascript should
     *   bind its calendar event handling to the return value of this method.
     */
    function getCalendar() {
      return $(_calendarId);
    };

    /**
     * @returns object
     *   An object containing the available availability settings.
     */
    function getSettings() {
      return _settings;
    };

    /**
     * @returns object
     *   An object containing the available availability states (indexed by the class).
     */
    function getStates() {
      return _settings.states;
    };

    /**
     * Returns the date range of the calendar.
     * @returns Object
     *   { from: Date, to: Date }
     */
    function getCalendarRange() {
      if (_calendarRange === null) {
        var from, to;
        // Get all tables representing calendar months within *this* calendar,
        // extract the month and update the from/to range
        $(_calendarId + ' table.calmonth').each(function() {
          var idParts = $(this).attr('id').split('-', 4);
          if (idParts.length == 4) {
            var year = idParts[1];
            var month = idParts[2];
            var calFrom = new Date(year, month - 1, 1);
            if (from === undefined || from > calFrom) {
              from = calFrom;
            }
            var calTo = new Date(year, month, 0); // 'They' say this works on all browsers (and should work according to the spec).
            if (to === undefined || to < calTo) {
              to = calTo;
            }
          }
        });
        _calendarRange = {from: from, to: to};
      }
      return _calendarRange;
    };

    /**
     * Returns whether the given date is within the calendar range
     * @param Date
     * @returns Boolean
     */
    function isInCalendarRange(date) {
      var range = getCalendarRange();
      return range.from <= date && date <= range.to;
    }

    /**
     * Returns the number of months the calendar displays.
     *
     * @returns Integer
     */
    function getNumberOfMonths() {
      var range = getCalendarRange();
      return (range.to.getFullYear() - range.from.getFullYear()) * 12
        + range.to.getMonth() - range.from.getMonth() + 1;
    }

    /**
     * Returns the state for a given cell.
     *
     * @param jQuery
     *   Table cell representing a day in the calendar to be changed.
     * @returns String|Object
     *   A string for a whole day state or an object for a split day state: { state: String, am: String, pm: String }
     */
    function _getCellState(cell) {
      var state = {am: '', pm: ''};
      // Loop through all states (not including inherited properties).
      for (var key in _settings.states) {
        if (_settings.states.hasOwnProperty(key)) {
          var cssClass = _settings.states[key].class;
          if (cell.hasClass(cssClass)) {
            // Distribute over am and pm but do not overwrite.
            state.am = state.am || cssClass;
            state.pm = state.pm || cssClass;
          }
          if (cell.hasClass(cssClass + '-am')) {
            state.am = cssClass;
          }
          if (cell.hasClass(cssClass + '-pm')) {
            state.pm = cssClass;
          }
        }
      }
      return state.am === state.pm ? state.am : state;
    }

    /**
     * Returns the state for a given date.
     *
     * @param Date date
     * @returns null|String|Object
     *   A string for a whole day state or an object for a
     *   split day state: { state: String, am: String, pm: String } or
     *   null for a date not within the calendar range.
     */
    function getDayState(date) {
      var cellState = null;
      if (isInCalendarRange(date)) {
        // Get table for the given month and drill down to the table cell representing the given date.
        $('#cal-' + date.toFormattedString('yyyy-mm') + '-' + _nid)
          .find('td')
          .filter(function() {
            return $(this).html().search(new RegExp('^<span>' + date.getDate() + '<', 'i')) === 0;
          })
          .each(function() {
            cellState = _getCellState($(this));
          });
      }
      return cellState;
    }

    /**
     * Checks whether all dates in the given range are available.
     * In the split day situation we check from params.from pm to params.to am.
     * In the whole day situation we check from params.from up to but not including params.to.
     *
     * @param Object params { from: Date, to: Date }
     * @returns Boolean|null null if the given date range is not fully within the calendar range.
     */
    function isAvailable(params) {
      var available = null;
      var to = params.to;
      if (!_settings.splitDay) {
        // We don't have to check for the last day itself.
        to = new Date(to.getFullYear(), to.getMonth(), to.getDate() - 1);
      }
      if (isInCalendarRange(params.from) && isInCalendarRange(to)) {
        available = true;
        var date = params.from;
        while (available && date <= to) {
          currentDayState = getDayState(date);
          if (typeof currentDayState == 'string') {
            available = getStates()[currentDayState].is_available;
          }
          else {
            if (date > params.from || !_settings.splitDay) {
              // am state is to be taken into account.
              available = getStates()[currentDayState.am].is_available;
            }
            if (date < params.from || !_settings.splitDay) {
              // pm state is to be taken into account.
              available = available && getStates()[currentDayState.pm].is_available;
            }
          }
        }
      }
      return available;
    }

    /**
     * @param object cell
     *   Table cell representing a day in the calendar to be changed.
     * @param String state
     * @param int dayPart
     *   Which part of the day: 1 = am, 2 = pm, 3 = whole day.
     *   Ignored if the calendar does not support split days.
     */
    function _setCellState(cell, state, dayPart) {
      // Ignore dayPart if calendar does not support split days.
      if (dayPart === undefined || !_settings.splitDay) {
        dayPart = 3;
      }
      if (_settings.states.hasOwnProperty(state)) {
        // 'Normal' availability state: remove original states, taking into account dayPart.
        for (var key in _settings.states) {
          if (_settings.states.hasOwnProperty(key)) {
            var cssClass = _settings.states[key].class;
            switch (dayPart) {
              case 3: // Whole day: remove original class.
                cell.removeClass(cssClass + '-am ' + cssClass + '-pm ' + cssClass);
                break;
              case 2: // PM
                if (cell.hasClass(cssClass)) {
                  // Replace original whole day class with original AM class.
                  cell.removeClass(cssClass).addClass(cssClass + '-am');
                }
                else if (cell.hasClass(cssClass + '-pm')) {
                  // Remove PM class.
                  cell.removeClass(cssClass + '-pm');
                }
                break;
              case 1: // AM
                if (cell.hasClass(cssClass)) {
                  // Replace original whole day class with original PM class.
                  cell.removeClass(cssClass).addClass(cssClass + '-pm');
                }
                else if (cell.hasClass(cssClass + '-am')) {
                  // Remove AM class.
                  cell.removeClass(cssClass + '-am');
                }
                break;
            }
          }
        }
        // Add new state
        cell.addClass(state + (dayPart == 3 ? '' : dayPart == 2 ? '-pm' : '-am'));
      }
      else {
        // 'Special' availability state: just add it, ignoring dayPart.
        cell.addClass(state);
      }
    };

    /**
     * Removes a state from the given cell.
     *
     * @param jQuery cell
     * @param String state
     * @param int dayPart
     */
    function _removeCellState(cell, state, dayPart) {
      if (dayPart === undefined || !_settings.splitDay) {
        dayPart = 3;
      }
      cell.removeClass(state + (dayPart == 3 ? '' : dayPart == 2 ? '-pm' : '-am'));
    };

    /**
     * Changes the state of the date to the given state.
     * If the calendar supports split days, the from date will be changed for pm only, and the
     * to date will be changed for am only.
     * If the state is one of the defined states, it will replace the existing state. If not
     * defined, it will be treated as a 'special' state (like calother, calpast and caltoday)
     * and just added to the day.
     *
     * @param Date date
     * @param String state
     * @param int dayPart
     *   Which part of the day: 1 = am, 2 = pm, 3 = whole day.
     *   Ignored if the calendar does not support split days.
     */
    function setDayState(date, state, dayPart) {
      // Get table for the given month and drill down to the table cell representing the given date.
      $('#cal-' + date.toFormattedString('yyyy-mm') + '-' + _nid)
        .find('td')
        .filter(function() {
          return $(this).html().search(new RegExp('^<span>' + date.getDate() + '<', 'i')) === 0;
        })
        .each(function() {
          _setCellState($(this), state, dayPart);
        });
    };

    /**
     * Sets all days in the from - to range to the given state. The range includes the from and
     * to dates themselves. If the calendar supports split days, the from date will be changed
     * for pm only, and the to date will be changed for am only.
     *
     * @param Object params
     *   { from: Date, to: Date, state: String }
     */
    function setDayStates(params) {
      // Start with PM, but if only setting 1 day, we set the whole day.
      var dayPart = params.from.valueOf() === params.to.valueOf() ? 3 : 2;
      // We can only set dates within the calendar range: ignore other dates.
      var calendarRange = getCalendarRange();
      if (calendarRange.from > params.from) {
        params.from = calendarRange.from;
      }
      if (calendarRange.to < params.to) {
        params.to = calendarRange.to;
      }
      var day = params.from;
      while (day <= params.to) {
        setDayState(day, params.state, dayPart);
        day = new Date(day.getFullYear(), day.getMonth(), day.getDate() + 1);
        dayPart = day >= params.to ? 1 : 3;
      };
    };

    /**
     * Removes a state from the given date.
     *
     * @param Date date
     * @param String state
     * @param int dayPart
     */
    function removeDayState(date, state, dayPart) {
      // Get table for the given month
      $('#cal-' + date.toFormattedString('yyyy-mm') + '-' + _nid)
      .find('td')
      .filter(function() {
        return $(this).html().search(new RegExp('^<span>' + date.getDate() + '<', 'i')) === 0;
      })
      .each(function() {
        _removeCellState($(this), state, dayPart);
      });
    };

    return {
      // Publicly exposed methods:
      getCalendar: getCalendar,
      getSettings: getSettings,
      getStates: getStates,
      getCalendarRange: getCalendarRange,
      isInCalendarRange: isInCalendarRange,
      getNumberOfMonths: getNumberOfMonths,
      getDayState: getDayState,
      isAvailable: isAvailable,
      setDayState: setDayState,
      setDayStates: setDayStates,
      removeDayState: removeDayState
    };
  });

  /**
   * Initializes the @see AvailabilityCalendar API object on load.
   */
  Drupal.behaviors.availabilityCalendars = {
    attach: function(context, settings) {
      Drupal.availabilityCalendars = new AvailabilityCalendars(settings.availabilityCalendars);
    }
  };
})(jQuery);
