/*
 * The code below serves as an example of how to enable and configure the viewport feature:
 *
 * Copy, paste and adapt the following code to your custom PHP code:
  // Add js script and settings
  drupal_add_js(array(
    'mysettings' => array(
      'settings' => array(
        'month' => array('width' => 222, 'height' => 241),
        'viewport' => array('cols' => 3, 'rows' => 2),
        'step' => array('cols' => 0, 'rows' => 1)
      ),
      'backwardSelector' => '#ac_up',
      'forwardSelector' => '#ac_down',
    )),
    array('type' => 'setting')
  );
  drupal_add_js(drupal_get_path('module', 'availability_calendars') .'/availability_calendars.view.js', array('type' => 'file', 'scope' => 'footer'));
  drupal_add_js("Drupal.behaviors.availabilityCalendarsViewport = {
    attach: function(context, settings) {
      Drupal.availabilityCalendarsViewport = new AvailabilityCalendarsViewport(Drupal.availabilityCalendars,
        settings.mysettings.settings,
        settings.mysettings.backwardSelector,
        settings.mysettings.forwardSelector
      );
    }
  };", array('type' => 'inline', 'scope' => 'footer'));
 *
 *
 * Additional styling may be applied to the .availability-calendars-viewport class, being the
 * outer viewport div. Typical settings to define are (but not limited to):
 * - width/height: if the width and height of 1 month are not passed in via the settings
 * - border
 * - margin
 * - background
 */

(function($) {
  /**
   * @class AvailibilityCalendarsViewport
   *
   * @constructor Creates a new AvailibilityCalendarsViewport object.
   * @param object calendar
   *   Required: the calendar on which to operate.
   * @param object settings
   *   Optional: object with the following properties
   *   (name type (default) description):
   *   {
   *     month
   *       width     int (0) the width of 1 calendar month, used to calculate the width of the
   *                         viewport. If 0, use CSS to define the width of the viewport.
   *       height    int (0) the height of 1 calendar month, used to calculate the height of the
   *                         viewport. If 0, use CSS to define the height of the viewport.
   *     viewport
   *       cols      int (3) The number of months that is shown horizontally in the viewport.
   *       rows      int (2) The number of months that is shown vertically in the viewport.
   *     step
   *       cols      int (0) indicating how many columns of months to move horizontally on stepping.
   *       rows      int (1) indicating how many rows of months to move vertically on stepping.
   *     animate     Animation parameters, @see http://api.jquery.com/animate/
   *       backward  Advanced usage, do not pass in.
   *       forward   Advanced usage, do not pass in.
   *       speed     int|string ('slow') The animation speed, see jquery documentation of animate().
   *     totalMonths int (calculated) Advanced usage, do not pass in.
   *     firstMonth  int (calculated) Advanced usage, do not pass in.
   *   }
   * @param mixed backwardSelector
   *   Optional: any string or object that can be fed to the jquery()
   *   constructor function to return a jquery object and that defines the elements the user can
   *   click on to scroll the calendar backward. Typically this will be a selector string.
   * @param mixed forwardSelector
   *   Optional: like backwardSelector but this parameter defines the
   *   elements to scroll the calendar forward.
   */
  // For the time being (as long as this is not enabled and configured via the UI) make this a
  // global, so the above example PHP code will work.
  //var AvailabilityCalendarsViewport = (function(calendar, settings, backwardSelector, forwardSelector) {
  AvailabilityCalendarsViewport = (function(calendar, settings, backwardSelector, forwardSelector) {
     // Initialize settings.
    settings = $.extend(true, {
      month: {width: 0, height: 0},
      viewport: {cols: 3, rows: 2},
      step: {cols: 0, rows: 1},
      animate: {speed: 'slow'},
      totalMonths: calendar.getNumberOfMonths(),
      firstMonth: 1
    }, settings);
    if (!settings.animate.backward) {
      settings.animate.backward = {
          top: '+=' + (settings.step.rows * settings.month.height),
          left: '+=' + (settings.step.cols * settings.month.width)
      };
    };
    if (!settings.animate.forward) {
      settings.animate.forward = {
          top: '-=' + (settings.step.rows * settings.month.height),
          left: '-=' + (settings.step.cols * settings.month.width)
        };
    };

    // Wrap the viepwort around the calendar months (not including the key).
    // - An outer div with overflow: hidden to obscure the non visible months.
    // - An inner div that is positioned relative to move the visble part forward and backward
    //   within the viewport.
    // - If scrolling horizontally we want the iner viewport to be infinitely wide.
    $('.calmonth-wrapper', calendar.getCalendar())
      .wrapAll('<div class="availability-calendars-viewport"><div class="availability-calendars-viewport-inner"></div></div>');
    // Initialize viewport dimensions (Only if passed in, f not passed in set this via CSS).
    var viewport = $('.availability-calendars-viewport', calendar.getCalendar());
    if (settings.month.width != 0) {
      viewport.width(settings.viewport.cols * settings.month.width);
    }
    if (settings.month.height != 0) {
      viewport.height(settings.viewport.rows * settings.month.height);
    }
    // Move to the inner viewport as that is the element to be animated.
    viewport = viewport.children();
    if (settings.step.cols != 0) {
      viewport.width(10000);
    }

    // Initialize the event handlers.
    if (backwardSelector) {
      $(backwardSelector).click(function() {
        scrollBackward();
      });
    }
    if (forwardSelector) {
      $(forwardSelector).click(function() {
        scrollForward();
      });
    }

    /**
     * Scroll the viewport backward (if possible).
     */
    function scrollBackward() {
      if (settings.firstMonth > 1) {
        viewport.animate(settings.animate.backward, settings.animate.speed);
        settings.firstMonth -= settings.step.rows * settings.viewport.cols + settings.step.cols * settings.viewport.rows;
        // @todo: enable/disable? (CSS3 pointer-events and CSS cursor properties?)
      }
    };

    /**
     * Scroll the viewport forward (if possible).
     */
    function scrollForward() {
      if (settings.firstMonth + settings.viewport.rows * settings.viewport.cols <= settings.totalMonths) {
        viewport.animate(settings.animate.forward, settings.animate.speed);
        settings.firstMonth += settings.step.rows * settings.viewport.cols + settings.step.cols * settings.viewport.rows;
        // @todo: enable/disable? (CSS3 pointer-events and CSS cursor properties?)
      }
    };

    return {
      // Publicly exposed methods:
      scrollBackward: scrollBackward,
      scrollForward: scrollForward
    };
  });
})(jQuery);
