(function($) {
/**
 * @class Drupal.availabilityCalendar.Command represents a calendar state changing
 * command during the whole creation phase, i.e. from click on a state to the click
 * on the end date.
 *
 * @todo: if we won't use a textarea anymore: remove the partial showing code.
 */
Drupal.availabilityCalendar.Command = function(cid) {
  this.state = "";
  this.from = null;
  this.to = null;
  this.elt = $("#availability-changes-" + cid);

  /**
   * Sets the state of the current command and, as this is supposed to be the first
   * parameter to be set, cleans the from and to dates.
   */
  this.setState = function(selectedState) {
    if (selectedState !== undefined) {
      this.state = selectedState;
    }
    this.from = null;
    this.to = null;
    this.show();
  };

  /**
   * Adds a date to the command. If it is the 1st date it will be the from date. If it is
   * the 2nd date it will be the to date, if necessary, swapping the from and to dates.
   *
   * @param Date date
   */
  this.addDate = function(date) {
    if (this.from === null) {
      this.from = date;
    }
    else if (this.to === null) {
      this.to = date;
      if (this.to < this.from) {
        var _from = this.from;
        this.from = this.to;
        this.to = _from;
      }
      if (Drupal.availabilityCalendar.get(cid).isSplitDay()) {
        // Split day: to date = departure date = am only: store as "from" to "to - 1 day".
        this.to.setDate(this.to.getDate() - 1);
        if (this.to < this.from) {
          // In the split day situation, clicking twice on the same day results
          // in an empty period: refuse.
          this.to = null;
        }
      }
    }
    this.show();
  };

  /**
   * @returns Boolean Whether the current command is complete.
   */
  this.isComplete = function() {
    return this.to !== null && this.from !== null & this.state !== '';
  };

  /**
   * Shows (and replaces) the current command in the accompanying textarea or hidden field.
   */
  this.show = function() {
    var val = this.elt.val();
    var pos = val.lastIndexOf("\n") + 1;
    val = val.substr(0, pos) + this.toString();
    this.elt.val(val);
  };

  /**
   * Finishes the current command and starts a new one.
   */
  this.finish = function() {
    this.show();
    this.elt.val(this.elt.val() + "\n");
    this.setState();
  };

  /**
   * @returns String
   *   A string representation of the current command.
   */
  this.toString = function() {
    result = "";
    result += "state: ";
    result += this.state !== "" ? this.state : "-";
    result += " from: ";
    result += this.from !== null ? this.from.toFormattedString("yyyy-mm-dd") : "-";
    result += " to: ";
    result += this.to !== null ? this.to.toFormattedString("yyyy-mm-dd") : "-";
    return result;
  };
};

/**
 * @class Drupal.availabilityCalendar.Edit provides the glueing code that connects the
 * form elements on entity edit forms (for entities with an availability calendar field)
 * with the @see AvailabilityCalendar API object and the
 * @see Drupal.availabilityCalendar.Command class.
 */
Drupal.availabilityCalendar.Edit = function(cid) {
  var calendar = Drupal.availabilityCalendar.get(cid);
  var command = new Drupal.availabilityCalendar.Command(cid);
  //var changed = false;

  // Initialize command.
  var formRadios = $(".form-radios.availability-states-" + cid);
  command.setState($("input:radio:checked", formRadios).val());
  // Add css_class of states as class to wrapper elements around the radios.
  $("input:radio", formRadios).parent().addClass(function() {
    return Drupal.availabilityCalendar.states.getClass($(this).children("input:radio:").val());
  });

  // Attach to state radios events.
  $("input:radio", formRadios).click(function() {
    // State clicked: remove calselected (if set) and restart current command
    if (command.from !== null) {
      calendar.removeExtraState(command.from, "cal-selected");
    }
    command.setState($("input:radio:checked", formRadios).val());
  });

  // Attach to the calendar events.
  calendar.getCalendar().bind("calendarclick", function(event, date, cid) {
    command.addDate(date);
    if (!command.isComplete()) {
      calendar.addExtraState(command.from, "cal-selected");
    }
    else {
      calendar.removeExtraState(command.from, "cal-selected");
      calendar.removeExtraState(command.to, "cal-selected");
      calendar.changeRangeState(command.from, command.to, command.state);
      command.finish();

      /* For now, we don't show the warning as it now is part of an entity edt form
      // Show warning.
      if (changed === false) {
        var warning = '<div class="messages warning"> ' + Drupal.t('Changes made to this calendar will not be saved until the form is submitted.') + '</div>';
        $(warning).insertBefore('#availability-calendar-node-edit-calendar-form .form-actions').hide().fadeIn('slow');
        changed = true;
      }
      */
    }
  });

  return null;
};
})(jQuery);
