(function($) {
/**
 * @class Drupal.availabilityCalendar.BookingFormlet provides the glueing code
 *   that connects the reservation formlet with the availability calendar fields
 *   on the page.
 * @param Integer bookedState
 *   The state id (sid) to visually change the state to after the user has
 *   clicked both an arrival and departure date. This gives the visitor visual
 *   feedback and suggest that the state changes to "optionally booked".
 * @see AvailabilityCalendars API object.
 */
Drupal.availabilityCalendar.BookingFormlet = function(bookedState) {
  var cid = null;
  var from = null;
  var to = null;

  init();
  show();

  function init() {
    // Attach to the calendar click events from all calendars.
    var calendars = Drupal.availabilityCalendar.getAll();
    for (var calendar in calendars) {
      if (calendars.hasOwnProperty(calendar)) {
        calendars[calendar].getCalendar().bind('calendarclick', addDate);
      }
    }

    // Attach to the click events of the reset buttons.
    $('#acbf-reset-from').click(resetFrom);
    $('#acbf-reset-both').click(resetBoth);
  }

  /**
   * Adds a date to the command.
   * - If it is the first date, it will be the from date.
   * - If it is the 2nd date, it will be the to date, swapping the from and to dates if needed.
   * - If it is a 3rd date, either the from or to date will be changed, depending on whether
   *   the 3rd date is before the current from or not.
   */
  function addDate(event, date, calendarCid) {
    if (cid !== calendarCid) {
      resetBoth();
      cid = calendarCid;
      $('#availability-calendar-booking-formlet-form input[name="cid"]').val(cid);
    }

    if (from === null) {
      from = date;
    }
    else {
      var shiftTo = true;
      if (to !== null) {
        Drupal.availabilityCalendar.get(cid).restoreRangeState(from, to);
        if (date < from) {
          from = date;
          shiftTo = false;
        }
        else {
          to = date;
        }
      }
      else {
        if (date >= from) {
          to = date;
        }
        else {
          to = from;
          from = date;
        }
      }

      if (Drupal.availabilityCalendar.get(cid).isSplitDay() && shiftTo) {
        // Split day: to date = departure date = am only: store as "from" to "to - 1 day".
        to.setDate(to.getDate() - 1);
        if (to < from) {
          // In the split day situation, clicking twice on the same day results
          // in an empty period: refuse.
          to = null;
        }
      }
    }
    show();
  };

  /**
   * Resets the from date and restores the calendar.
   */
  function resetFrom() {
    if (cid !== null) {
      if (from !== null && to === null) {
        Drupal.availabilityCalendar.get(cid).removeExtraState(from, 'cal-selected');
      }
      from = null;
      show();
    }
    return false;
  };

  /**
   * Resets both dates and restores the calendar.
   */
  function resetBoth() {
    if (cid !== null) {
      if (from !== null && to !== null) {
        Drupal.availabilityCalendar.get(cid).restoreRangeState(from, to);
      }
      from = to = null;
      show();
    }
    return false;
  }

  /**
   * @returns Boolean Whether the current form values are valid.
   */
  function isValid() {
    return cid !== null && to !== null && from !== null;
  };

  /**
   * Shows the current values, help texts, and enables the submit button.
   */
  function show() {
    if (from === null) {
      $('.form-reset').css('display', 'none');
      $('#edit-arrival')
        .attr('disabled', 'disabled')
        .val(Drupal.t('Click on an available date in the calendar'));
      $('#edit-departure')
        .val('')
        .attr('disabled', 'disabled');
    }
    else {
      $('#edit-arrival')
        .val(from.toFormattedString('dd-mm-yyyy'))
        .removeAttr('disabled');
      if (to === null) {
        $('#acbf-reset-from').css('display', 'inline-block');
        $('#acbf-reset-both').css('display', 'none');
        $('#edit-departure')
          .val(Drupal.t('Click on your departure date'))
          .attr('disabled', 'disabled');
        Drupal.availabilityCalendar.get(cid).addExtraState(from, 'cal-selected');
      }
      else {
        var showTo = new Date(to.getFullYear(), to.getMonth(), to.getDate() + 1);
        $('#edit-departure')
          .val(showTo.toFormattedString('dd-mm-yyyy'))
          .removeAttr('disabled');
        $('#acbf-reset-from').css('display', 'none');
        $('#acbf-reset-both').css('display', 'inline-block');
        // If to < from then calselected was set on to date, so clear both.
        Drupal.availabilityCalendar.get(cid).removeExtraState(from, 'cal-selected');
        Drupal.availabilityCalendar.get(cid).removeExtraState(showTo, 'cal-selected');
        if (isValid()) {
          Drupal.availabilityCalendar.get(cid).changeRangeState(from, to, bookedState);
        }
      }
    }
    if (isValid()) {
      $('#edit-submit').removeAttr('disabled');
    }
    else {
      $('#edit-submit').attr('disabled', 'disabled');
    }

  };

  // Nothing to return (no public interface)
  return true;
};
})(jQuery);
