<?php
/**
 * @file
 * Views support for Availability Calendar.
 *
 * Availability Calendar supports the views module. As this is not a normal
 * field, we need to implement some views hooks and classes to get it to work.
 * Basically we provide:
 * - Fields:
 *   - Availability Calendar: because this is a field this works without custom
 *     code, whether the view displays content or fields.
 *   - Created date and changed date: though the current formatters do not show
 *     this information, you can visualise them with a view.
 * - (Exposed) Filter criteria and arguments:
 *   - Enabled: only if there's a bundle that allows to override it on a per
 *     entity basis.
 *   - Created date.
 *   - Changed date.
 *   - Available.
 * - Sort criteria:
 *   - Created date.
 *   - Changed date.
 *
 * We also hide some things in the views UI:
 * - Argument/Filter:
 *   - Cid value.
 *   - Enabled value: if there's no bundle that allows to override it per node.
 * - Sort:
 *   - Cid value.
 *   - Enabled value.
 * - All revisioning options as basically our field is not revisioned.
 *
 * @author Erwin Derksen (http://drupal.org/user/750928)
 */

/**
 * Implements hook_field_views_data_alter().
 *
 * The data structure contains too much non-descriptive information for me.
 * so we let views create the array for us first, then we alter it
 * (as opposed to implementing hook_field_views_data() itself).
 */
function availability_calendar_field_views_data_alter(&$data, $field, $module) {
  if ($field['type'] == 'availability_calendar') {
    // Get some info for easy use later on.
    $field_name = $field['field_name'];
    $field_table_name = key($field['storage']['details']['sql']['FIELD_LOAD_CURRENT']);
    $revision_table_name = key($field['storage']['details']['sql']['FIELD_LOAD_REVISION']);
    $entity_types = array_keys($field['bundles']);
    $field_views_info = $data[$field_table_name][$field_name];
    $field_title = $field_views_info['title'];
    $join_extra = array(
      array(
        'table' => $field_table_name,
        'field' => "{$field_name}_enabled",
        'value' => 1,
        'numeric' => TRUE,
      ),
    );

    // Availability calendar field is not (really) revisioned: remove.
    unset($data[$revision_table_name]);

    // field_..._cid is not interesting in itself. Change it to use as filter
    // on availability:
    // - Make the join an INNER join (I don't understand how Views can use LEFT
    //     as default when used in filters).
    // - Change UI texts.
    // - Change argument handler.
    // - Change filter handler.
    // - Remove sort handler.
    unset($data[$field_table_name]["{$field_name}_cid"]['sort']);
    foreach ($data[$field_table_name]['table']['join'] as $entity_type => &$join) {
      $join['type'] = 'INNER';
      // We should only join the availability_calendar_availability table when
      // the calendar is enabled: we add this extra join condition.
      $join['extra'] = $join_extra;
    }
    $data[$field_table_name]["{$field_name}_cid"]['title'] = t('@field_label available', array('@field_label' => $field_title));
    $data[$field_table_name]["{$field_name}_cid"]['title short'] = t('@field_label available', array('@field_label' => $field_title));
    $data[$field_table_name]["{$field_name}_cid"]['help'] = t('Filters on availability during the defined period.') . ' ' . $field_views_info['help'];
    $data[$field_table_name]["{$field_name}_cid"]['argument']['handler'] = 'views_handler_argument_date';
    $data[$field_table_name]["{$field_name}_cid"]['filter']['handler'] = 'availability_calendar_handler_filter_availability';
    $data[$field_table_name]["{$field_name}_cid"]['filter']['default_state'] = $field['settings']['default_state'];

    // field_..._enabled: remove as sort. Argument and filter should be there
    // for some (administrative) edge use cases. However, note that as soon as
    // content of one of the availability_calendar_* tables is used, the join
    // condition will contain an extra filter on enabled.
    unset($data[$field_table_name]["{$field_name}_enabled"]['sort']);
    $data[$field_table_name]["{$field_name}_enabled"]['title'] = t('@field_label enabled', array('@field_label' => $field_title));
    $data[$field_table_name]["{$field_name}_enabled"]['title short'] = t('@field_label enabled', array('@field_label' => $field_title));
    $data[$field_table_name]["{$field_name}_enabled"]['help'] = t('Whether the calendar has been enabled', array('@field_label' => $field_title));
    $data[$field_table_name]["{$field_name}_enabled"]['filter']['handler'] = 'views_handler_filter_boolean_operator';

    // Expose table availability_calendar_calendar with fields created and changed
    $table_availability_calendar_calendar = array();
    foreach ($entity_types as $entity_type) {
      $table_availability_calendar_calendar['table']['join'][$entity_type] = array(
        'type' => 'INNER',
        'left_table' => $field_table_name,
        'left_field' => "{$field_name}_cid",
        'field' => 'cid',
        // We should only join the availability_calendar_calendar table when
        // the calendar is enabled: we add this extra join condition.
        'extra' => $join_extra,
      );
    }
    $table_availability_calendar_calendar['created'] = array(
      'group' => $field_views_info['group'],
      'title' => t('@field_label created date', array('@field_label' => $field_title)),
      'title short' => t('@field_label created', array('@field_label' => $field_title)),
      'help' => t('The date the calendar was created.') . ' ' . $field_views_info['help'],
      'field' => array(
        'table' => 'availability_calendar_calendar',
        'handler' => 'views_handler_field_date',
        'click sortable' => true,
        'field_name' => 'created',
        'element type' => 'span',
      ),
      'argument' => array(
        'field' => 'created',
        'table' => 'availability_calendar_calendar',
        'handler' => 'views_handler_argument_date',
        'field_name' => 'created',
      ),
      'filter' => array(
        'field' => 'created',
        'table' => 'availability_calendar_calendar',
        'handler' => 'views_handler_filter_date',
        'field_name' => 'created',
        'allow empty' => FALSE,
      ),
      'sort' => array(
        'field' => 'created',
        'table' => 'availability_calendar_calendar',
        'handler' => 'views_handler_sort',
        'field_name' => 'created',
      ),
    );
    $table_availability_calendar_calendar['changed'] = array(
      'group' => $field_views_info['group'],
      'title' => t('@field_label updated date', array('@field_label' => $field_title)),
      'title short' => t('@field_label updated', array('@field_label' => $field_title)),
      'help' => t('The date the calendar was last updated.') . ' ' . $field_views_info['help'],
      'field' => array(
        'table' => 'availability_calendar_calendar',
        'handler' => 'views_handler_field_date',
        'click sortable' => true,
        'field_name' => 'changed',
        'element type' => 'span',
      ),
      'argument' => array(
        'field' => 'changed',
        'table' => 'availability_calendar_calendar',
        'handler' => 'views_handler_argument_date',
        'field_name' => 'changed',
      ),
      'filter' => array(
        'field' => 'changed',
        'table' => 'availability_calendar_calendar',
        'handler' => 'views_handler_filter_date',
        'field_name' => 'changed',
        'allow empty' => FALSE,
      ),
      'sort' => array(
        'field' => 'changed',
        'table' => 'availability_calendar_calendar',
        'handler' => 'views_handler_sort',
        'field_name' => 'changed',
      ),
    );
    $data['availability_calendar_calendar'] = $table_availability_calendar_calendar;
  }
}
