<?php

/**
 * @class availability_calendar_handler_filter_availability Views handler to
 * filter on availability.
 *
 * This filter inherits from @see views_handler_filter_date. But we make it more
 * specific to:
 * - Accept dates only (not DateTime's).
 * - Not accepting relative dates.
 * - Accept only future dates.
 * - Allow only 2 operators:
 *   - =: available at a given date
 *   - between: available from a given departure date to a departure date
 * - Add date popups to the (exposed) form elements.
 */
class availability_calendar_handler_filter_availability extends views_handler_filter_date {
  public static $instance;

  function __construct() {
    self::$instance = $this;
  }

  function option_definition() {
    $options = parent::option_definition();

    $options['operator'] = array('default' => 'between');
    $options['exposed'] = array('default' => TRUE);

    return $options;
  }

  function operators() {
    // We only allow "available at <date>" and
    // "available between <from> and <to>" as operators.
    $operators = array(
      '=' => array(
        'title' => t('At (date)'),
        'method' => 'op_simple',
        'short' => t('at'),
        'values' => 1,
      ),
      'between' => array(
        'title' => t('Period'),
        'method' => 'op_between',
        'short' => t('between'),
        'values' => 2,
      ),
    );

    return $operators;
  }

  /**
   * Add validation and date popup(s) to the value form.
   */
  function value_form(&$form, &$form_state) {
    parent::value_form($form, $form_state);
    // Remove the option to define dates relatively that date filter (our parent
    // class) added.
    if (empty($form_state['exposed']) && isset($form['value']['type'])) {
      $form['value']['type']['#type'] = 'hidden';
      $form['value']['type']['#default_value'] = 'date';
      $form['value']['type']['#access'] = FALSE;  // Don't send to the client.
    }

    // Add validation.
    $form['value']['#element_validate'][] = 'availability_calendar_handler_filter_availability_validate_value';

    // Use date popups if that module is available.
    if (module_exists('date_popup')) {
      if (isset($form['value']['min'])) {
        $form['value']['min'] = $this->change_element_into_date_popup($form['value']['min'], t('Arrival date'));
        $form['value']['max'] = $this->change_element_into_date_popup($form['value']['max'], t('Departure date'));
      }
      else if (isset($form['value']['#type']) && $form['value']['#type'] == 'textfield') {
        $form['value'] = $this->change_element_into_date_popup($form['value'], '');
      }
    }
  }

  /**
   * Changes a (text) form element into a dete popup element.
   *
   * @param array $element
   * @return array
   *   The changed form element.
   */
  protected function change_element_into_date_popup($element, $title) {
    $element['#type'] = 'date_popup';
    if (!empty($title)) {
      $element['#title'] = $title;
      $element['#date_label_position'] = '';
    }
    $element['#date_type'] = 'DATE_ISO';
    $element['#date_format'] = 'Y-m-d';
    $element['#date_year_range'] = '-0:+2';
    availability_calendar_handler_filter_availability_js_alter($dummy = TRUE);
    return $element;
  }

  /**
   * Validate that the time values convert to something usable.
   */
  function validate_valid_time(&$form, $operator, $value) {
    $operators = $this->operators();

    $required = FALSE;
    if ($operator === NULL) {
      // We are in exposed form validate.
      $required = (bool) $this->options['expose']['required'];
      $operator = array_key_exists('min', $form) ? 'between' : '=';
    }

    $now = new DateTime();
    $now = $now->modify('-1 day')->format(AC_ISODATE);
    if ($operators[$operator]['values'] == 1) {
      $this->validate_valid_time_1($form, $value, $required, $now, t('Only future availability can be searched.'));
    }
    elseif ($operators[$operator]['values'] == 2) {
      $min_valid = $this->validate_valid_time_1($form['min'], $value['min'], $required, $now, t('Only future availability can be searched.'));
      $this->validate_valid_time_1($form['max'], $value['max'], $required || !empty($value['min']), $min_valid ? $value['min'] : NULL, t('The departure date should be after the arrival date.'));
    }
  }

  protected function validate_valid_time_1(&$element, $value, $required, $minimum, $minimum_error_message) {
    $valid = TRUE;
    // Note that the value can be an array with a date and time component.
    if (is_array($value)) {
      $value = $value['date'];
    }
    if (empty($value)) {
      if ($required) {
        form_error($element, t('Field %field is required.', array('%field' => $element['#title'])));
        $valid = FALSE;
      }
    }
    else if (!checkdate(substr($value, 5, 2), substr($value, 8, 2), substr($value, 0, 4))) {
      form_error($element, t('Invalid date format.'));
      $valid = FALSE;
    }
    else if (!empty($minimum) && $value <= $minimum) {
      form_error($element, $minimum_error_message);
      $valid = FALSE;
    }
    return $valid;
  }

  function op_between($field) {
    module_load_include('inc', 'availability_calendar', 'availability_calendar.widget');
    availability_calendar_query_available($this->query, $this->table_alias, $this->field, new DateTime($this->value['min']), new DateTime($this->value['max']), $this->definition['default_state']);
  }

  function op_simple($field) {
    $this->value['min'] = $this->value['value'];
    $this->value['max'] = new DateTime($this->value['min']);
    $this->value['max'] = $this->value['max']->modify('+1 day')->format(AC_ISODATE);
    return $this->op_between($field);
  }
}

/**
 * Form element validator for the date field(s): forward to the
 * @see availability_calendar_handler_filter_availability::validate_valid_time()
 * method inside the class.
 */
function availability_calendar_handler_filter_availability_validate_value($element, &$form_state, $form) {
  availability_calendar_handler_filter_availability::$instance->validate_valid_time($element, NULL, $form_state['values'][$element['#parents'][0]]);
}

/**
 * Called by hook_alter_js().
 *
 * This function changes the date popups added by the class above.
 */
function availability_calendar_handler_filter_availability_js_alter(&$javascript) {
  static $adapt = FALSE;
  if ($javascript === TRUE) {
    $adapt = TRUE;
    return;
  }
  if ($adapt && isset($javascript['settings']['data'])) {
    foreach ($javascript['settings']['data'] as &$setting) {
      if (is_array($setting) && isset($setting['datePopup'])) {
        foreach ($setting['datePopup'] as &$date_popup_settings) {
          $date_popup_settings['settings']['minDate'] = 0;
        }
      }
    }
  }
}
