(function ($) {
  "use strict";

  // This code may appear on pages without any calendar, thus without any
  // other js from this module: assure Drupal.availabilityCalendar is defined.
  Drupal.availabilityCalendar = Drupal.availabilityCalendar || {};

  /**
   * @param {HTMLInputElement} dateField1
   * @param {HTMLInputElement} dateField2
   */
  Drupal.availabilityCalendar.dateRangePicker = function (dateField1, dateField2) {
    // Couple the 2 inputs via data attributes each pointing to the other input.
    // This will turn the datepicker into a date range picker. 
    $(dateField1).attr('data-date-range-end', "#" + dateField2.id);
    $(dateField2).attr('data-date-range-start', "#" + dateField1.id);

    // Unbind focus events from 2nd date as may have been set by date_popup.js:
    // only way to do so is by unbinding ALL focus handlers.
    $(dateField2).unbind("focus");

    // And set our own date range picker focus handler on the 2nd date field. 
    $(dateField2).bind("focus click", function (event, fromHere) {
      if (!fromHere) {
        // When the 2nd date gets focus or gets clicked, show the date range
        // picker that is attached to the 1st date.
        $(dateField1).focus();
        // But we still want the actual focus on the 2nd date to allow manually
        // entering or clearing a value. Prevent recursion by passing true for
        // the parameter fromHere.
        $(this).trigger("focus", [true]);
      }
    });
  };

  /**
   * Initialization code based on the Drupal behaviors paradigm.
   *
   * If the form uses auto submit, we prevent that it does so on coupled
   * elements that together define a date range, by adding the
   * ctools-auto-submit-exclude class to the date inputs.
   * However, that means that we have to take that task over by triggering the
   * auto submit if both elements have a (correct) value.
   */
  Drupal.behaviors.availabilityCalendarDateRangePickerAutoSubmit = {
    attach: function (context, settings) {
      var emptyDatesSent = true;

      // Copied (and adapted) from ctools/js/auto-submit.js.
      function triggerSubmit(form) {
        form = $(form);
        if (!form.hasClass("ctools-ajaxing")) {
          form.find(".ctools-auto-submit-click").click();
        }
      }

      function validate(value, format) {
        try {
          return Boolean($.datepicker.parseDate(format, value));
        }
        catch (e) {
          return false;
        }
      }

      $(".ctools-auto-submit-full-form", context)
        .find("[data-date-range-end], [data-date-range-start]")
        .filter(":not(.ctools-auto-submit-exclude)")
        .addClass("ctools-auto-submit-exclude date-range-picker-auto-submit")
        .bind("change", function (e) {
          if ($(e.target).is(":not(.date-range-picker-auto-submit-exclude)")) {
            var elt = $(this);
            var other = $(elt.data("data-date-range-end") || elt.data("data-date-range-start"));
            if (elt.val() === "" && other.val() === "") {
              if (!emptyDatesSent) {
                emptyDatesSent = true;
                triggerSubmit(this.form);
              }
            }
            else {
              var format = settings.datePopup && settings.datePopup.hasOwnProperty(this.id) ? settings.datePopup[this.id].dateFormat : "";
              if (validate(elt.val(), format) && validate(other.val(), format)) {
                emptyDatesSent = false;
                triggerSubmit(this.form);
              }
            }
          }
        });
    }
  };

}(jQuery));
