<?php

namespace Drupal\avif;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\Exception\FileException;
use Drupal\Core\File\FileSystem;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Image\ImageFactory;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;

/**
 * Class Avif.
 *
 * @package Drupal\avif
 */
class Avif {

  use StringTranslationTrait;

  /**
   * The image factory.
   *
   * @var \Drupal\Core\Image\ImageFactory
   */
  protected $imageFactory;

  /**
   * Logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Default image processing quality.
   *
   * @var int
   */
  protected $defaultQuality;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystem
   */
  protected $fileSystem;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\Client
   */
  protected $httpClient;

  /**
   * Avif conversion service endpoint.
   *
   * @var int
   */
  protected $endpoint;

  /**
   * Avif constructor.
   *
   * @param \Drupal\Core\Image\ImageFactory $imageFactory
   *   Image factory to be used.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   Logger channel factory.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $stringTranslation
   *   String translation interface.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Configuration factory.
   * @param \Drupal\Core\File\FileSystem $fileSystem
   *   The file system service.
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client.
   */
  public function __construct(ImageFactory $imageFactory, LoggerChannelFactoryInterface $loggerFactory, TranslationInterface $stringTranslation, ConfigFactoryInterface $configFactory, FileSystem $fileSystem, ClientInterface $httpClient) {
    $this->imageFactory = $imageFactory;
    $this->logger = $loggerFactory->get('avif');
    $this->setStringTranslation($stringTranslation);
    $this->defaultQuality = $configFactory->get('avif.settings')->get('quality');
    $this->fileSystem = $fileSystem;
    $this->httpClient = $httpClient;
    $this->endpoint = $configFactory->get('avif.settings')->get('endpoint');
  }

  /**
   * Creates a Avif copy of a source image URI.
   *
   * @param string $uri
   *   Image URI.
   * @param int $quality
   *   Image quality factor (optional).
   *
   * @return bool|string
   *   The location of the Avif image if successful, FALSE if not successful.
   */
  public function createAvifCopy($uri, $quality = NULL) {

    if (is_null($quality)) {
      $quality = $this->defaultQuality;
    }

    return $this->createAvifImage($uri, $quality);
  }

  /**
   * Deletes all image style derivatives.
   */
  public function deleteImageStyleDerivatives() {
    // Remove the styles directory and generated images.
    try {
      $this->fileSystem->deleteRecursive(file_default_scheme() . '://styles');
    }
    catch (FileException $e){
      $this->logger->error($e->getMessage());
      $error = $this->t('Could not delete image style directory while uninstalling Avif. You have to delete it manually.');
      $this->logger->error($error);
    }
  }

  /**
   * Receives the srcset string of an image and returns the avif equivalent.
   *
   * @param $srcset
   *   Srcset to convert to .avif version
   *
   * @return string
   *   Avif version of srcset
   */
  public function getAvifSrcset($srcset) {
    return preg_replace('/\.(png|jpg|jpeg)(\\?.*?)?(,| |$)/i', '.avif\\2\\3', $srcset);
  }

  /**
   * Creates a Avif copy of a source image URI using imagemagick toolkit.
   *
   * @param string $uri
   *   Image URI.
   * @param int $quality
   *   Image quality factor (optional).
   *
   * @return bool|string
   *   The location of the Avif image if successful, FALSE if not successful.
   */
  private function createAvifImage($uri, $quality = NULL) {
    $avifGenerated = $url = FALSE;

    try {
      if ($wrapper = \Drupal::service('stream_wrapper_manager')->getViaUri($uri)) {
        $url= $wrapper->getExternalUrl();
      }

      $response = $this->httpClient->get($this->endpoint, [
        'query' => [
          'url' => $url,
          'format' => 'avif',
          'quality' => $quality
        ]
      ]);

      if ($response->getStatusCode() == 200) {
        $pathInfo = pathinfo($uri);
        $destination = $pathInfo['dirname'] . '/' . $pathInfo['filename'] . '.avif';
        $avifGenerated = $this->fileSystem
        ->saveData($response->getBody(), $destination, FileSystemInterface::EXISTS_REPLACE);
      }
      else{
        $this->logger->error('The Avif conversion service returns this code error: %error.', ['%error' => $response->getStatusCode()]);
      }
    }
    catch (RequestException $e) {
      $this->logger->error('Failed to download Avif image using the external service.it due to "%error".', ['%error' => $e->getMessage()]);
    }

    return $avifGenerated;
  }

}
