<?php

namespace Drupal\avif\Plugin\AvifProcessor;

use Drupal\avif\Plugin\AvifProcessorBase;
use Drupal\Core\Image\ImageFactory;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Uses the Image Magick to convert images.
 *
 * @AvifProcessor(
 *   id = "imagemagick",
 *   label = @Translation("ImageMagick")
 * )
 */
class ImageMagick extends AvifProcessorBase implements ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The image factory.
   *
   * @var ImageFactory
   */
  protected $imageFactory;

  /**
   * Logger.
   *
   * @var LoggerInterface
   */
  protected $logger;

  /**
   * ImageMagick constructor.
   *
   * @param array $configuration
   * @param $plugin_id
   * @param $plugin_definition
   * @param ImageFactory $imageFactory
   *   Image factory to be used.
   * @param LoggerInterface $loggerFactory
   *   Logger channel factory.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ImageFactory $imageFactory, LoggerInterface $loggerFactory) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->imageFactory = $imageFactory;
    $this->logger = $loggerFactory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('image.factory'),
      $container->get('logger.factory')->get('avif')
    );
  }

  /**
   * Convert
   *
   * @param $image_uri
   * @param $quality
   * @param $destination
   * @return bool
   */
  public function convert($image_uri, $quality, $destination) {
    $avif = FALSE;

    // Check AVIF support
    $toolkit = $this->imageFactory->getToolkitId();
    if (!in_array('AVIF', $this->imageFactory->getSupportedExtensions($toolkit))) {
      $error = $this->t("Your image toolkit '%toolkit' doesn't support AVIF format.", ['%toolkit' => $toolkit]);
      $this->logger->error($error);

      return $avif;
    }

    if ($toolkit != 'imagemagick') {
      $error = $this->t("This plugin works only with Image Magick or your toolkit is: %toolkit.", ['%toolkit' => $toolkit]);
      $this->logger->error($error);

      return $avif;
    }

    $ImageMagickImg = $this->imageFactory->get($image_uri, 'imagemagick');

    // We'll convert the image into avif.
    $ImageMagickImg->apply('convert', [
      'extension' => 'avif',
      'quality' => $quality
    ]);

    $pathInfo = pathinfo($image_uri);
    $destination = $pathInfo['dirname'] . '/' . $pathInfo['filename'] . '.avif';
    if ($ImageMagickImg->save($pathInfo['dirname'] . '/' . $pathInfo['filename'] . '.avif')) {
      $avif = $destination;
    }
    else {
      $error = $this->t('Imagemagick issue: Could not generate Avif image.');
      $this->logger->error($error);
    }

    return $avif;
  }

}
