<?php

namespace Drupal\aws_sqs_api;

/**
 * A queue item retrieved from SQS.
 *
 * There are three required public properties:
 * - data
 * - item_id
 * - created.
 *
 * Additional methods are provided for AWS integration.
 *
 * @see \Drupal\Core\Queue\QueueInterface::claimItem()
 */
class AwsSqsQueueItem {

  /**
   * A reference to the queue which generated this item.
   *
   * This allows the item to be released/deleted/extended.
   *
   * @var \Drupal\aws_sqs_api\Queue\AwsSqsQueue
   */
  protected $queue;

  /**
   * Payload submitted to AWS.
   *
   * @var mixed
   */
  protected $data;

  /**
   * Unique reference for this queue entry, generated by AWS.
   *
   * @var string
   */
  // phpcs:ignore
  protected $item_id;

  /**
   * Timestamp when this entry was added to the queue (unix-epoch in seconds).
   *
   * @var int
   */
  protected $created;

  /**
   * Message group ID, which determines sequencing and guaranteed ordering.
   *
   * Applies to FIFO queue items only.
   *
   * @var string
   */
  protected $messageGroupId;

  /**
   * Receipt ID, which allows acknowledgement of process completion.
   *
   * @var string
   */
  protected $receipt;

  /**
   * Constructor requires the 3 mandatory fields.
   *
   * @param mixed $data
   *   Payload submitted to AWS.
   * @param string $item_id
   *   Unique reference for this queue entry, generated by AWS.
   * @param int $created
   *   Timestamp when this entry was added to the queue.
   */
  public function __construct($data, $item_id, $created) {
    $this->data = $data;
    $this->item_id = $item_id;
    $this->created = $created;
  }

  /**
   * Attach a reference to the queue, so that the item can be released.
   *
   * @param \Drupal\aws_sqs_api\Queue\AwsSqsQueue $queue
   *   The queue which generated this item.
   *
   * @return $this
   *   Fluid composition.
   */
  public function setQueue(AwsSqsQueue $queue) {
    $this->queue = $queue;
    return $this;
  }

  /**
   * Set the receipt ID so the item can be released or completed.
   *
   * @var string $receipt
   *   The unique receipt ID for receiving this message. AWS will generate a
   *   different receipt ID for each claim request.
   *
   * @return $this
   *   Fluid composition.
   */
  public function setReceipt($receipt) {
    $this->receipt = $receipt;
    return $this;
  }

  /**
   * For reliable queues, provide a group ID so that messages are in-order.
   *
   * @var string $messageGroupId
   *   The group ID provided by AWS.
   *
   * @return $this
   *   Fluid composition.
   */
  public function setMessageGroupId($messageGroupId) {
    $this->messageGroupId = $messageGroupId;
    return $this;
  }

  /**
   * Release this queue item for reprocessing by another worker.
   */
  public function release() {
    $this->queue->releaseItem($this);
  }

  /**
   * Delete this queue item.
   */
  public function delete() {
    $this->queue->deleteItem($this);
  }

  /**
   * Extend the lease of this item for an additional period of time.
   *
   * @param int $additional_time
   *   Additional time in seconds. The total lease time may not be more than
   *   the maximum available lease time of 43200 seconds (12 hours).
   */
  public function extendLease($additional_time) {

  }

  /**
   * Magic getter function to return properties.
   */
  public function __get($property) {
    if (property_exists($this, $property)) {
      return $this->$property;
    }
  }

  /**
   * Magic function to determine if a property is set.
   */
  public function __isset($property) {
    return property_exists($this, $property) && !empty($this->$property);
  }

}
