<?php

namespace Drupal\azure_ad\Form;

use Drupal;
use Drupal\azure_ad\Helper\moAzureADHelper;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\user\Entity\User;
use Drupal\user_provisioning\moUserProvisioningConstants;
use Exception;
use Drupal\user_provisioning\Helpers\moUserProvisioningLogger;
use Drupal\user_provisioning\moUserProvisioningOperationsHandler;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Response;

class MoDrupalToAzureSync extends FormBase
{
    private ImmutableConfig $config;
    private Config $config_factory;
    protected $messenger;
    private moUserProvisioningLogger $mo_logger;
    private $base_url;

    public function __construct(){
        global $base_url;
        $this->config = Drupal::config('azure_ad.settings');
        $this->config_factory = Drupal::configFactory()->getEditable('azure_ad.settings');
        $this->messenger = Drupal::messenger();
        $this->mo_logger = new moUserProvisioningLogger();
        $this->base_url = $base_url;
    }

    public function getFormId()
    {
        return "mo_drupal_to_azure";
    }

    public function buildForm(array $form, FormStateInterface $form_state)
    {
        $form['mo_azure_ad_overview_add_css'] = [
            '#attached' => [
                'library' => [
                    'azure_ad/azure_ad.admin',
                ]
            ],
        ];

        $form['azure_ad_summary_header_style'] = [
            '#markup' => t('<div class="mo_azure_header_container_step1">'),
        ];

        $form['mo_azure_ad_manual_provisioning_back_to_overview'] = [
            '#type' => 'submit',
            '#value' => t('&#11164; &nbsp;Back to Provisioning types'),
            '#button_type' => 'danger',
            '#submit' => ['::moAzureProvBackStep3Button'],
        ];

        $this->AzureADSummary($form, $form_state);
        $status = $this->config->get('azure_ad_provisioning_step');

        if ($status == 'auto_configuration') {
            $this->AzureADAutoProvisioning($form, $form_state);
        } else if ($status == 'manual_configuration'){
            $this->AzureADManualProvisioning($form, $form_state);
        } else if ($status == 'scheduler_configuration'){
            $this->AzureADSchedulerProvisioning($form, $form_state);
        } else {
            $this->AzureADOverviewTab($form, $form_state);
        }

        $azure_ad_helper = new moAzureADHelper();
        $azure_ad_helper->moAzureShowCustomerSupportIcon($form, $form_state);

        return $form;
    }

    public function submitForm(array &$form, FormStateInterface $form_state){}

    public function AzureADSummary(array &$form, FormStateInterface $form_state){

        $form['azure_ad_summary_details'] = [
            '#type' => 'details',
            '#title' => $this->t('Summary'),
        ];

        $overview = new MoAzureOverview();
        $overview->AzureADConfigurationTable($form, $form_state, 'summary');

        $form['azure_ad_summary_details']['configure_azure_save_button'] = [
            '#type' => 'submit',
            '#value' => t('Save Configuration'),
            '#button_type' => 'primary',
            '#submit' => ['::saveAzureADSummary'],
        ];

        $overview->moAzureTestResult($form, $form_state);
    }

    private function AzureADAutoProvisioning(array &$form, FormStateInterface $form_state){

        $form['mo_azure_ad_provisioning_fieldset'] = [
            '#type' => 'fieldset',
            '#title' => t('Automatic Provisioning Configuration<hr>'),
        ];

        $form['mo_azure_ad_provisioning_fieldset']['automatic_provisioning_choose_operations'] = [
            '#type' => 'fieldset',
            '#title' => t('Choose Operations in Automatic Provisioning<hr>'),
        ];

        $form['mo_azure_ad_provisioning_fieldset']['automatic_provisioning_choose_operations']['azure_ad_provisioning_operations'] = [
            '#type' => 'table',
            '#responsive' => true,
        ];

        $row = $this->moProvisioningOperations('automatic');
        $form['mo_azure_ad_provisioning_fieldset']['automatic_provisioning_choose_operations']['azure_ad_provisioning_operations']['operations'] = $row;

        $this->attributeMapping($form,$form_state);
    }

    private function AzureADSchedulerProvisioning(array &$form, FormStateInterface $form_state){

        $form['mo_azure_ad_provisioning_fieldset'] = [
            '#type' => 'fieldset',
            '#title' => t('Scheduler Based Provisioning Configuration <a href="azure_upgrade_plans">[PREMIUM]</a><hr>'),
        ];

        $form['mo_azure_ad_provisioning_fieldset']['scheduler_provisioning_choose_operations'] = [
            '#type' => 'fieldset',
            '#title' => t('Choose Operations in Scheduler Based Provisioning<hr>'),
        ];

        $form['mo_azure_ad_provisioning_fieldset']['scheduler_provisioning_choose_operations']['azure_ad_provisioning_operations'] = [
            '#type' => 'table',
            '#responsive' => true,
        ];

        $row = $this->moProvisioningOperations('scheduler');
        $form['mo_azure_ad_provisioning_fieldset']['scheduler_provisioning_choose_operations']['azure_ad_provisioning_operations']['operations'] = $row;

        $this->attributeMapping($form,$form_state);
    }

    private function AzureADManualProvisioning(array &$form, FormStateInterface $form_state){

        $form['mo_azure_ad_manual_provisioning_back_to_overview'] = [
            '#type' => 'submit',
            '#value' => t('&#11164; &nbsp;Back to Provisioning types'),
            '#button_type' => 'danger',
            '#submit' => ['::moAzureProvBackStep3Button'],
        ];

        $form['mo_azure_ad_manual_provisioning_fieldset'] = [
            '#type' => 'fieldset',
            '#title' => t('Manual Provisioning Configuration<hr>'),
        ];

        $form['mo_azure_ad_manual_provisioning_fieldset']['manual_provisioning_choose_operations'] = [
            '#type' => 'fieldset',
            '#title' => t('Choose Operations in Manual Provisioning<hr>'),
        ];

        $form['mo_azure_ad_manual_provisioning_fieldset']['manual_provisioning_choose_operations']['azure_ad_provisioning_operations'] = [
            '#type' => 'table',
            '#responsive' => TRUE ,
        ];

        $row = $this->moProvisioningOperations('manual');
        $form['mo_azure_ad_manual_provisioning_fieldset']['manual_provisioning_choose_operations']['azure_ad_provisioning_operations']['operations'] = $row;

        $form['mo_azure_ad_manual_provisioning_fieldset']['manual_provisioning_choose_operations']['create_user_fieldset']=[
            '#type' => 'fieldset',
            '#title' => $this->t('Sync Users<hr>'),
            '#states' => [
                'visible' => [
                    ':input[name="azure_ad_provisioning_operations[operations][create_user_azure_ad]"]' => ['checked' => TRUE],
                ],
            ],
        ];

        $form['mo_azure_ad_manual_provisioning_fieldset']['manual_provisioning_choose_operations']['create_user_fieldset']['mo_azure_drupal_username'] = [
            '#type' => 'textfield',
            '#attributes' => ['placeholder' => 'Search Drupal username of user to sync'],
            '#autocomplete_route_name' => 'user_provisioning.autocomplete',
            '#prefix' => '<p class="mo_azure_highlight_background"><strong>Note:</strong> Search the username of user to sync (create) it to the Active Directory.</p><div class="container-inline">',
        ];

        $form['mo_azure_ad_manual_provisioning_fieldset']['manual_provisioning_choose_operations']['create_user_fieldset']['mo_azure_sync_button'] = [
            '#type' => 'submit',
            '#value' => t('Sync'),
            '#button_type' => 'primary',
            '#attributes' => ['class' => ['mo_azure_sync_button']],
            '#submit' => ['::moAzureManualSync'],
        ];

        $form['mo_azure_ad_manual_provisioning_fieldset']['manual_provisioning_choose_operations']['create_user_fieldset']['mo_azure_sync_all_button'] = [
            '#type' => 'submit',
            '#value' => t('Sync All Users'),
            '#button_type' => 'primary',
            '#attributes' => ['class' => ['mo_azure_sync_all_button']],
            '#disabled' => true,
        ];

        $this->attributeMapping($form,$form_state);
    }

    private function moProvisioningOperations($provision_type): array
    {
        $row['read_user_azure_ad'] = [
            '#type' => 'checkbox',
            '#title' => t('Read user'),
            '#default_value' => $provision_type != 'scheduler',
            '#disabled' => true,
            '#prefix'=> '<div class="container-inline">',
        ];

        $row['create_user_azure_ad'] = [
            '#type' => 'checkbox',
            '#title' => t('Create user '),
            '#default_value' => $provision_type == 'scheduler' ? false : $this->config->get('azure_ad_'.$provision_type.'_provisioning_checkbox'),
            '#disabled' => $provision_type == 'scheduler',
        ];

        $row['update_user_azure_ad'] = [
            '#type' => 'checkbox',
            '#title' => t('Update user <a><small>[PREMIUM]</small></a>'),
            '#disabled' => TRUE,
        ];

        $row['deactivate_user_azure_ad'] = [
            '#type' => 'checkbox',
            '#title' => t('Deactivate user <a><small>[PREMIUM]</small></a>'),
            '#disabled' => TRUE,
        ];

        $row['delete_user_azure_ad'] = [
            '#type' => 'checkbox',
            '#title' => t('Delete user <a><small>[PREMIUM]</small></a>'),
            '#disabled' => TRUE,
        ];

        $row['manual_provisioning_save_button'] = [
            '#type' => 'submit',
            '#value' => t('Save'),
            '#button_type' => 'primary',
            '#disabled' => $provision_type == 'scheduler',
            '#submit' => $provision_type == 'manual' ? ['::saveSettingsAzureADManualProv'] : ['::saveSettingsAzureADAutomaticProv'],
            '#suffix' => '</div>',
        ];

        return $row;
    }

    private function attributeMapping(array &$form, FormStateInterface $form_state){

        $form['mo_azure_attribute_mapping_details'] =[
            '#type' => 'details',
            '#open' => false,
            '#title' => t('Mapping <a href="azure_upgrade_plans"><small>[PREMIUM]</small></a>'),
        ];

        $form['mo_azure_attribute_mapping_details']['azure_ad_attribute_mapping'] = [
            '#type' => 'fieldset',
            '#title' => t('Basic Attribute Mapping<hr>'),
        ];

        $form['mo_azure_attribute_mapping_details']['azure_ad_attribute_mapping']['azure_ad_attribute_mapping_table'] = [
            '#type' => 'table',
            '#responsive' => TRUE ,
            '#attributes' => ['class' => ['attribute_mapping_table']]
        ];

        $row = $this->moAzureADAttributeMapTable();
        $form['mo_azure_attribute_mapping_details']['azure_ad_attribute_mapping']['azure_ad_attribute_mapping_table']['attributes'] = $row;

        $form['mo_azure_attribute_mapping_details']['azure_ad_attribute_mapping']['azure_ad_basic_mapping_save_button'] = [
            '#type' => 'submit',
            '#value' => t('Save Configuration '),
            '#button_type' => 'primary',
            '#disabled' => true,
        ];

        $form['mo_azure_attribute_mapping_details']['azure_ad_custom_attribute_mapping'] = array(
            '#type' => 'fieldset',
            '#title' => t('Custom Attribute Mapping<hr>'),
        );

        $form['mo_azure_attribute_mapping_details']['azure_ad_custom_attribute_mapping']['attribute_mapping_info'] = array(
            '#markup' => '<div class="mo_azure_highlight_background">This feature allows you to map the user attributes from your Drupal to Azure AD.</div>',
        );

        $custom_fields = [];
        $usr = User::load(\Drupal::currentUser()->id());
        $usrVal = $usr->toArray();
        foreach ($usrVal as $key => $value) {
            $custom_fields[$key] = $key;
        }

        $form['mo_azure_attribute_mapping_details']['azure_ad_custom_attribute_mapping']['azure_ad_custom_attribute_mapping_table'] = [
            '#type' => 'table',
            '#responsive' => TRUE ,
            '#attributes' => ['class' => ['custom_attribute_mapping_table']]
        ];

        $row = $this->moAzureADCustomAttributeMapTable($custom_fields);
        $form['mo_azure_attribute_mapping_details']['azure_ad_custom_attribute_mapping']['azure_ad_custom_attribute_mapping_table']['custom_mapping'] = $row;
    }

    private function moAzureADAttributeMapTable(){
        $row['username_attribute'] = [
            '#type' => 'select',
            '#title' => t('Username Attribute'),
            '#options' => [ 1 => 'name'],
            '#disabled' => true,
        ];

        $row['email_attribute'] = [
            '#type' => 'select',
            '#title' => t('Email Attribute'),
            '#options' => [ 1 => 'mail'],
            '#disabled' => true,
        ];

        return $row;
    }

    private function moAzureADCustomAttributeMapTable($custom_fields){
        $row['azure_ad_drupal_attr_name'] = [
            '#title' => t('Drupal Attribute Name'),
            '#type' => 'select',
            '#options' => $custom_fields,
            '#disabled' => true,
        ];

        $row['azure_ad_attribute_name'] = [
            '#type' => 'textfield',
            '#title' => t('Azure AD Attribute Name'),
            '#disabled' => true,
        ];

        $row['azure_ad_add_button'] = [
            '#type' => 'submit',
            '#button_type' => 'primary',
            '#disabled' => true,
            '#value' => '+',
        ];

        $row['azure_ad_sub_button'] = [
            '#type' => 'submit',
            '#button_type' => 'danger',
            '#disabled' => true,
            '#value' => '-',
        ];

        return $row;
    }

    public function saveAzureADSummary(array &$form, FormStateInterface $form_state){
        $overview = new MoAzureOverview();
        $overview->moAzureStep2SaveButton($form, $form_state);
    }

    public function saveSettingsAzureADManualProv(array &$form, FormStateInterface $form_state)
    {
        $this->config_factory->set('azure_ad_manual_provisioning_checkbox', $form_state->getValues()['azure_ad_provisioning_operations']['operations']['create_user_azure_ad'])->save();
        $this->messenger->addstatus(t('Configurations saved successfully.'));
    }

    public function saveSettingsAzureADAutomaticProv(array &$form, FormStateInterface $form_state)
    {
        $this->config_factory->set('azure_ad_automatic_provisioning_checkbox', $form_state->getValues()['azure_ad_provisioning_operations']['operations']['create_user_azure_ad'])->save();
        $this->messenger->addstatus(t('Configurations saved successfully.'));
    }

    public function moAzureManualSync(array &$form, FormStateInterface $form_state){
        $entity_name = $form_state->getValues()['mo_azure_drupal_username'];

        $this->mo_logger->addLog('Provisioning on demand. Entered entity name is :' . $entity_name, __LINE__, __FUNCTION__, basename(__FILE__));

        $entity = $this->getEntityToProvision($entity_name);
        if (is_null($entity)) {
            $this->messenger->addError('"' . $entity_name . '"' . $this->t(' can not be recognized as a valid User or Role. Enter a valid entity( User or Role) to provision.'));
        }

        $mo_entity_handler = new moUserProvisioningOperationsHandler($entity);
        try {
            $result = $mo_entity_handler->insert();
            if (is_null($result)) {
                $this->messenger->addError($this->t('An error occurred while provisioning the user, please refer to <a href="' . $this->base_url . moUserProvisioningConstants::DRUPAL_LOGS_PATH . '">drupal logs</a> for more information.'));
                return;
            }
            $this->messenger->addMessage($this->t($entity->label() . ' successfully created at the configured application.'));
        } catch (Exception $exception) {
            $this->messenger->addError($exception->getMessage());
        }
    }

    private function getEntityToProvision(string $entity_name) {
        $user = user_load_by_name($entity_name);
        if ($user != FALSE) {
            return User::load($user->id());
        }
        return NULL;
    }

    private function AzureADOverviewTab(array &$form, FormStateInterface $form_state){
        $response = new RedirectResponse(Url::fromRoute('azure_ad.overview')->toString());
        $response->send();
        return new Response();
    }

    public function moAzureProvBackStep3Button(){
        $this->config_factory->set('azure_ad_provisioning_step', 'step3')->save();
        $response = new RedirectResponse(Url::fromRoute('azure_ad.overview')->toString());
        $response->send();
        return new Response();
    }

}