<?php

namespace Drupal\azure_ai_services\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for Azure OpenAI settings.
 */
class AzureOpenAISettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'azure_openai_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'azure_openai.config',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('azure_openai.config');

    // Connection settings
    // https://learn.microsoft.com/en-us/azure/ai-services/openai/reference
    $form['connection'] = [
      '#type' => 'details',
      '#title' => t('Connection settings'),
      '#description' => t('Connection settings used to connect to Azure OpenAI.'),
      '#open' => TRUE,
    ];

    $form['connection']['resource_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Resource Name'),
      '#default_value' => $config->get('resource_name') ? $config->get('resource_name') : '',
      '#required' => TRUE,
    ];

    $form['connection']['deployment_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Deployment ID'),
      '#default_value' => $config->get('deployment_id') ? $config->get('deployment_id') : '',
      '#required' => TRUE,
    ];

    $form['connection']['api_version'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Version'),
      '#default_value' => $config->get('api_version') ? $config->get('api_version') : '',
      '#required' => TRUE,
    ];

    $stored_key = $config->get('api_key');
    $masked_placeholder = $stored_key ? str_repeat('*', strlen($stored_key)) : '';

    $form['connection']['api_key'] = [
      '#type' => 'password',
      '#title' => $this->t('API Key'),
      '#description' => $stored_key ? $this->t('Enter a new key to replace the existing one. Leave blank to keep the current key.') : '',
      '#default_value' => '',
      '#required' => !$stored_key,
      '#attributes' => [
        'placeholder' => $masked_placeholder,
        'autocomplete' => 'off',
      ],
    ];

    // Prompt settings.
    $form['prompt'] = [
      '#type' => 'details',
      '#title' => t('Prompt settings'),
      '#description' => t('Prompt settings to optimize OpenAI responses.'),
      '#open' => TRUE,
    ];

    $form['prompt']['system_context'] = [
      '#type' => 'textarea',
      '#title' => $this->t('System Context'),
      '#default_value' => $config->get('system_context') ? $config->get('system_context') : '',
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $this->config('azure_openai.config')
      ->set('resource_name', $form_state->getValue('resource_name'))
      ->set('deployment_id', $form_state->getValue('deployment_id'))
      ->set('api_version', $form_state->getValue('api_version'))
      ->set('system_context', $form_state->getValue('system_context'))
      ->save();

    $new_key = $form_state->getValue(['connection', 'api_key']);
    if (!empty($new_key)) {
      $this->config('azure_openai.config')
        ->set('api_key', $new_key)
        ->save();
    }
  }

}
