<?php

namespace Drupal\azure_ai_services\Service;

use GuzzleHttp\Client;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Service for calling Azure OpenAI chat completion endpoint.
 */
class AzureOpenAIService {

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs the AzureOpenAIService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   */
  public function __construct(ConfigFactoryInterface $configFactory) {
    $this->configFactory = $configFactory;
  }

  /**
   * Makes a request to the Azure OpenAI Chat Completion API.
   *
   * @param string $user_prompt
   *   The user prompt.
   * @param string|null $context
   *   Optional additional context to append to the system message.
   *
   * @return string|null
   *   The AI-generated response, or NULL on failure.
   */
  public function makeAzureOpenaiChatCompletion($user_prompt, $context = NULL) {
    // Fetch Azure OpenAI configuration from Drupal configuration.
    $config = $this->configFactory->get('azure_openai.config');
    $resource_name = $config->get('resource_name');
    $deployment_id = $config->get('deployment_id');
    $api_version = $config->get('api_version');
    $api_key = $config->get('api_key');
    $system_context = $config->get('system_context');

    if (!is_null($context)) {
      $system_context = $system_context . " " . $context;
    }

    $endpoint = 'https://' . $resource_name . '.openai.azure.com/openai/deployments/' . $deployment_id . '/chat/completions?api-version=' . $api_version;

    $client = new Client();

    // Prepare the request data.
    $request_data = [
      'messages' => [
        ["role" => "assistant", "content" => $system_context],
        ["role" => "user", "content" => $user_prompt],
      ],
      'max_tokens' => 4096,
    ];

    $headers = [
      'Content-Type' => 'application/json',
      'api-key' => $api_key,
    ];

    try {
      $response = $client->post($endpoint, [
        'headers' => $headers,
        'json' => $request_data,
        'verify' => FALSE,
      ]);

      $response_body = $response->getBody()->getContents();
      $response = json_decode($response_body, TRUE)['choices'][0]['message']['content'];

      return $response;
    }
    catch (\Exception $e) {
      // Handle error gracefully.
      \Drupal::logger('azure_openai')->error('Azure OpenAI request failed: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

}
