<?php

namespace Drupal\azure_blob_fs;

use Drupal\azure_blob_fs\PathProcessor\AzureBlobFsPathProcessorImageStyles;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\DependencyInjection\ServiceProviderBase;
use Drupal\Core\Site\Settings;
use Drupal\azure_blob_fs\StreamWrapper\PrivateAzureBlobFsStream;
use Drupal\azure_blob_fs\StreamWrapper\PublicAzureBlobFsStream;
use Drupal\azure_blob_fs\Asset\AzureBlobFsCssOptimizer;

/**
 * The stream wrapper class.
 *
 * In the docs for this class, anywhere you see "<scheme>", it can mean either
 * "azure_blob" or "public", depending on which stream is currently being serviced.
 */
class AzureBlobFsServiceProvider extends ServiceProviderBase {

  /**
   * Modifies existing service definitions.
   *
   * @param ContainerBuilder $container
   *   The ContainerBuilder whose service definitions can be altered.
   */
  public function alter(ContainerBuilder $container): void {
    // For now, we're doing a wide service override.
    // When the module is enabled, the core file system is replaced.
    // @TODO - In the future, we want to NOT do this and have the Azure file system available alongside the local filesystem.
    // @TODO - Additionally, we will want a PUBLIC AZURE file system and a PRIVATE AZURE file system to differentiate the two.
    // @TODO - Again, we don't really want to decorate/override core service, but instead add the azure file system to make it available for configurations on fields and in the file system site config.
    if (Settings::get('azure_blob_fs.enabled')
      && !empty(Settings::get('azure_blob_fs.sas_token'))
      && !empty(Settings::get('azure_blob_fs.blob_storage_url'))
      && !empty(Settings::get('azure_blob_fs.container_name'))) {

      // Get the default local public stream wrapper.
      $corePublicStreamWrapper = $container->getDefinition('stream_wrapper.public');
      $corePublicStreamWrapperClass = $corePublicStreamWrapper->getClass();

      // Register the core public stream wrapper as a separate one.
      // We will preserve it under the name 'local'.
      $container->register('stream_wrapper.local', $corePublicStreamWrapperClass)
                ->addTag('stream_wrapper', ['scheme' => 'local']);

      // Replace the core public stream wrapper with our Azure wrapper.
      $container->getDefinition('stream_wrapper.public')
        ->setClass(PublicAzureBlobFsStream::class);

      // Replace the core Image Style Path Processor with ours.
      $container->getDefinition('path_processor.image_styles')
        ->setClass(AzureBlobFsPathProcessorImageStyles::class);

      // Fix CSS static urls.
      $container->getDefinition('asset.css.optimizer')
        ->setClass(AzureBlobFsCssOptimizer::class);

      if ($container->hasDefinition('stream_wrapper.private')) {
        // Replace the private stream wrapper with our Azure wrapper.
        $container->getDefinition('stream_wrapper.private')
          ->setClass(PrivateAzureBlobFsStream::class);
      }
    }
  }

}
