<?php

namespace Drupal\azure_blob_fs\File;

use Drupal\azure_blob_fs\Service\AzureBlobFsService;
use Drupal\Core\File\Exception\DirectoryNotReadyException;
use Drupal\Core\File\Exception\FileException;
use Drupal\Core\File\Exception\FileExistsException;
use Drupal\Core\File\Exception\FileNotExistsException;
use Drupal\Core\File\FileSystem;
use Drupal\Core\Site\Settings;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;
use Psr\Log\LoggerInterface;

/**
 * Provides a FileSystem service decorator for the Azure Blob File System.
 */
class AzureBlobFsFileSystem extends FileSystem {

  /**
   * Original service object.
   *
   * @var \Drupal\Core\File\FileSystem $file_system
   */
  protected $fileSystem;

  /**
   * The module's main service.
   *
   * @var \Drupal\azure_blob_fs\Service\AzureBlobFsService
   */
  protected $azureBlobFsService;

  /**
   * Constructs a new FileSystem to decorate the original.
   *
   * @param \Drupal\Core\File\FileSystem $file_system
   *   The original FileSystem core service.
   * @param \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface $stream_wrapper_manager
   *   The stream wrapper manager.
   * @param \Drupal\Core\Site\Settings $settings
   *   The site settings.
   * @param \Psr\Log\LoggerInterface $logger
   *   The file logger channel.
   * @param \Drupal\azure_blob_fs\Service\AzureBlobFsService $azureBlobFsService
   *   Azure Blob Fs main service.
   */
  public function __construct(FileSystem $file_system, StreamWrapperManagerInterface $stream_wrapper_manager, Settings $settings, LoggerInterface $logger, AzureBlobFsService $azureBlobFsService) {
    $this->fileSystem = $file_system;
    $this->azureBlobFsService = $azureBlobFsService;
    parent::__construct($stream_wrapper_manager, $settings, $logger);
  }

  /**
   * I'm just fed up at this point.
   *
   * This is copy pasted from the core file system, but with a minor change.
   *
   * We don't want to append the basename of the source file to the destination.
   *
   * @see \Drupal\Core\File\FileSystem
   *
   * {@inheritdoc}
   */
  protected function prepareDestination($source, &$destination, $replace): void {
    // If our connection isn't wired, simply do default behaviour.
    if (!$this->azureBlobFsService->wired()) {
      parent::prepareDestination($source, $destination, $replace);
      return;
    }
  }

  /**
   * In Azure Blob Storages, directories don't exist and don't need to be
   * prepared.
   *
   * We want this to return true regardless.
   *
   * {@inheritdoc}
   */
  public function prepareDirectory(&$directory, $options = self::MODIFY_PERMISSIONS) {
    // If our connection isn't wired, simply do default behaviour.
    if (!$this->azureBlobFsService->wired()) {
      return parent::prepareDirectory($directory, $options);
    }

    // Otherwise, just return TRUE.
    return TRUE;
  }

}

