<?php

namespace Drupal\azure_content_moderator;

use Drupal\azure_content_moderator\Form\AzureContentModeratorSettingsForm;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Logger\LoggerChannelFactory;
use Drupal\wsdata\WSDataService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Fetch and format books.
 */
class AzureContentModeratorService {
  /**
   * Config Factory for Drupal.
   *
   * @var \Drupal\Core\Config\ConfigFactors
   */
  protected ConfigFactory $configFactory;

  /**
   * Service for WSData used in caching.
   *
   * @var \Drupal\wsdata\WSDataService
   */
  protected WSDataService $wsdata;

  /**
   * Service for logging messages.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactory
   */
  protected LoggerChannelFactory $loggerFactory;

  /**
   * Constructor to create object.
   *
   * @param \Drupal\Core\Config\ConfigFactory $configFactory
   *   Config factory to get azure profanity config.
   * @param \Drupal\wsdata\WSDataService $wsdata
   *   Service used for caching.
   * @param \Drupal\Core\Logger\LoggerChannelFactory $loggerFactory
   *   Logger factory used to log messages.
   */
  public function __construct(ConfigFactory $configFactory, WSDataService $wsdata, LoggerChannelFactory $loggerFactory) {
    $this->configFactory = $configFactory;
    $this->wsdata = $wsdata;
    $this->loggerFactory = $loggerFactory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('http_client'),
      $container->get('config.factory'),
      $container->get('wsdata'),
      $container->get('logger.factory'),
    );
  }

  /**
   * Returns the results from the profanity check.
   *
   *   Render array.
   */
  public function getResults($text = '') {
    if (empty($text)) {
      return;
    }

    // Grab the parameters from the config form.
    $config = $this->configFactory->get(AzureContentModeratorSettingsForm::CONFIG_NAME);
    $blockListNames = $config->get('block_list_names');
    $haltOnBlockListHit = $config->get('halt_on_block_list_hit');
    $apiVersion = $config->get('api_version');

    if (isset($blockListNames) === FALSE ||
       isset($haltOnBlockListHit) === FALSE ||
       empty($apiVersion)) {
      $this->loggerFactory->get('azure_content_moderator')->alert('A required parameter for the azure content moderation api is null.');
      return;
    }

    // Convert the form data to a format we can send to the API.
    // Format the input for the API.
    $categories = $config->get('categories');
    $categories = array_filter($categories, function($category) {
      return $category == 1;
    });

    if (count($categories) > 0) {
      // If no categories are set then we default to using all categories.
      // The API defaults to using all categories if we do not set that value.
      $data['categories'] = $categories;
    }

    $replacement = [
      'apiVersion' => $apiVersion,
    ];

    $data = [
      'text' => $text,
      'blocklistNames' => explode(',', $blockListNames),
      'haltOnBlocklistHit' => boolval($haltOnBlockListHit),
    ];

    $results = $this->wsdata->call('azure_content_moderation_service', NULL, $replacement, $data);

    return $results;
  }

  /**
   * Returns the results from the profanity check.
   *
   *   Render array.
   */
  public function isAppropriate($results) {
    if ($this->isValidParams($results)) {
      $this->loggerFactory->get('azure_profanity')->alert('The input for azure profanity check is invalid.');
      return;
    }

    $config = $this->configFactory->get(AzureContentModeratorSettingsForm::CONFIG_NAME);
    $categoriesAnalysis = $results['categoriesAnalysis'];
    $isValid = count($results['blocklistsMatch']) === 0;

    $tolerance = [
      "Hate" => intval($config->get('hate_tolerance_category')),
      "SelfHarm" => intval($config->get('hate_tolerance_category')),
      "Sexual" => intval($config->get('hate_tolerance_category')),
      "Violence" => intval($config->get('hate_tolerance_category')),
    ];

    foreach ($categoriesAnalysis as $item) {
      if ($item['severity'] >= $tolerance[$item["category"]]) {
        $isValid = FALSE;
        break;
      }
    }

    return $isValid;
  }

  public function toString($results) {
    if ($this->isValidParams($results)) {
      $this->loggerFactory->get('azure_content_moderator')->alert('The input for azure profanity check is invalid.');
      return;
    }

    return  json_encode($results);
  }

  private function isValidParams($results) {
    return ((empty($results) === TRUE ||
      array_key_exists('blocklistsMatch', $results) === FALSE ||
      array_key_exists('categoriesAnalysis', $results) === FALSE));
  }
}
