<?php

namespace Drupal\b24_webform\Form;

use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Render\Renderer;
use Drupal\webform\Entity\Webform;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\b24\Service\RestManager;

/**
 * Defines a form for configuring webform submissions export to Bitrix24.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Drupal\b24\Service\RestManager definition.
   *
   * @var \Drupal\b24\Service\RestManager
   */
  protected RestManager $b24RestManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The HTML renderer.
   *
   * @var \Drupal\Core\Render\Renderer
   */
  protected Renderer $renderer;

  /**
   * Constructs a new SettingsForm object.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    RestManager $b24_rest_manager,
    EntityTypeManagerInterface $entity_type_manager,
    Renderer $renderer
  ) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->b24RestManager = $b24_rest_manager;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('b24.rest_manager'),
      $container->get('entity_type.manager'),
      $container->get('renderer'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'b24_webform.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form_state->setCached(FALSE);

    $webforms = $this->entityTypeManager->getStorage('webform')->loadMultiple();
    $options = [];
    foreach ($webforms as $webform_id => $webform) {
      $options[$webform_id] = $webform->label();
    }

    $form['webform_select'] = [
      '#type' => 'select',
      '#title' => $this->t('Choose webform'),
      '#description' => $this->t('Choose webform to configure.'),
      '#options' => $options,
      '#empty_option' => $this->t('- Select -'),
      '#ajax' => [
        'callback' => '::getAjaxWebformMappingForm',
        'event' => 'change',
        'wrapper' => 'webform_mapping_form_wrapper',
      ],
    ];

    $form['webforms'] = $this->getWebformMappingForm($form, $form_state);

    return parent::buildForm($form, $form_state);
  }

  /**
   * Generate form elements for configure the chosen webform export to Bitrix24.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form current state of the form.
   *
   * @return array
   *   Array of form elements to configure parameters of the chosen webform.
   */
  public function getWebformMappingForm(
    array $form,
    FormStateInterface $form_state
  ) {
    $form['webforms'] = [
      '#tree' => TRUE,
      '#type' => 'fieldset',
      '#title' => $this->t('Webform submissions exporting rules'),
      '#prefix' => '<div id="webform_mapping_form_wrapper">',
      '#suffix' => '</div>',
      '#description' => $this->t('No webform chosen'),
    ];
    $webform_storage = $this->entityTypeManager->getStorage('webform');
    $webforms = $webform_storage->loadMultiple();
    $config = $this->config('b24_webform.settings');
    $chosen_webform = $form_state->getValue($form_state->getUserInput()['_triggering_element_name'] ?? NULL);
    if (!$chosen_webform) {
      $form['webforms']['#description'] = $this->t('No webform chosen');
      return $form['webforms'];
    }

    unset($form['webforms']['#description']);
    $webform = empty($form_state->getUserInput()['_triggering_element_name']) ? reset($webforms) : $webform_storage->load($chosen_webform);
    $webform_settings = $config->get($webform->id());

    $form['webforms'][$webform->id()] = [
      '#type' => 'fieldset',
      '#title' => $webform->label(),
    ];

    $form['webforms'][$webform->id()]['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable export'),
      '#description' => $this->t('Export submissions of this webform to Bitrix24 leads'),
      '#default_value' => $webform_settings['status'] ?? 0,
      '#description_display' => 'after',
    ];

    $form['webforms'][$webform->id()]['mapping'] = [
      '#type' => 'details',
      '#title' => $this->t('Fields mapping'),
      '#states' => [
        'visible' => [
          '[name="webforms[' . $webform->id() . '][status]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form_state->setRebuild(TRUE);
    $this->getFields($form['webforms'][$webform->id()]['mapping'], $webform);
    return $form['webforms'];
  }

  /**
   * Ajax callback for 'webform_select' element.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The current state of the form.
   *
   * @return array
   *   The webforms list form element array.
   */
  public function getAjaxWebformMappingForm(
    array &$form,
    FormStateInterface $formState
  ) {
    $formState->setRebuild();
    return $form['webforms'];
  }

  /**
   * Return configuration fields according to those provided by Bitrix24.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\webform\Entity\Webform $webform
   *   The currently processed webform.
   *
   * @return array|mixed
   *   An array of Bitrix 24 fields.
   */
  private function getFields(array &$form, Webform $webform) {
    $config = $this->config("b24_webform.settings")->get($webform->id());
    $b24_fields = $this->b24RestManager->getFields('lead');

    $elements = $webform->getElementsDecodedAndFlattened();
    $restricted_types = [
      'webform_actions',
    ];
    $options = [];
    foreach ($elements as $element_id => $element) {
      if (!isset($element['#type']) || in_array($element['#type'],
          $restricted_types) || !isset($element['#title'])) {
        continue;
      }
      $options[$element_id] = $element['#title'];
    }
    $options['custom'] = $this->t('Custom...');
    $token_link = [
      '#theme' => 'token_tree_link',
      '#token_types' => ['site', 'webform', 'webform_submission'],
      '#show_restricted' => TRUE,
      '#global_types' => TRUE,
    ];
    $token_link = $this->renderer->render($token_link);
    foreach ($b24_fields as $b24_field_name => $b24_field_properties) {
      if (in_array($b24_field_properties['type'],
          [
            'string',
            'crm_multifield',
            'double',
          ]
        ) && !$b24_field_properties['isReadOnly']) {
        $form[$b24_field_name] = [
          '#type' => 'select',
          '#options' => $options,
          '#title' => $b24_field_properties['title'],
          '#default_value' => $config['mapping'][$b24_field_name] ?? '',
          '#empty_option' => $this->t('- None -'),
        ];
        if ($b24_field_properties['isRequired']) {
          $form[$b24_field_name]['#label_attributes']['class'][] = 'form-required';
        }

        $form["{$b24_field_name}_custom"] = [
          '#type' => 'textarea',
          '#title' => $this->t('@name custom value',
            ['@name' => $b24_field_properties['title']]),
          '#description' => $token_link,
          '#description_display' => 'after',
          '#states' => [
            'visible' => [
              '[name="webforms[' . $webform->id() . '][mapping][' . $b24_field_name . ']"]' => [
                'value' => 'custom',
              ],
            ],
          ],
          '#default_value' => $config['mapping']["{$b24_field_name}_custom"] ?? '',
        ];
      }
    }

    return $b24_fields;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
    $form_state->cleanValues();
    $values = $form_state->getValues();
    if (!isset($values['webforms'])) {
      return;
    }
    foreach ($values['webforms'] as $webform_id => $webform_values) {
      if ($webform_values['status']) {
        foreach ($webform_values['mapping'] as $key => $value) {
          if (!$value && isset($form['webforms'][$webform_id]['mapping'][$key]['#label_attributes']['class']) && in_array('form-required',
              $form['webforms'][$webform_id]['mapping'][$key]['#label_attributes']['class'])) {
            $form_state->setErrorByName("webforms][$webform_id][mapping][$key",
              $this->t('@name field is required.',
                ['@name' => '«' . $form['webforms'][$webform_id]['mapping'][$key]['#title'] . '»']));
          }
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    $form_state->cleanValues();
    $config = $this->config('b24_webform.settings');
    foreach ($form_state->getValue('webforms') as $webform_id => $webform_settings) {
      $config->set($webform_id, $webform_settings);
    }
    $config->save();
  }

}
