<?php

declare(strict_types=1);

namespace Drupal\b24\Service;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityInterface;

/**
 * Handles references between Bitrix24 and Drupal entities.
 */
class ReferenceManager {

  /**
   * Constructs a ReferenceManager object.
   */
  public function __construct(
    private readonly Connection $database,
  ) {}

  /**
   * Gets the reference between Drupal and Bitrix24 entities.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   A Drupal entity earlier exported to Bitrix24.
   * @param string $ext_type
   *   Bitrix24 entity machine_name.
   *
   * @return mixed
   *   The reference.
   */
  public function getReference(EntityInterface $entity, string $ext_type): mixed {
    return $this->database->select('b24_reference', 'b')
      ->fields('b', ['ext_id', 'hash'])
      ->condition('bundle', $entity->bundle())
      ->condition('ext_type', $ext_type)
      ->condition('entity_id', $entity->id())
      ->execute()->fetchAssoc();
  }

  /**
   * Inserts new b24 reference to a database.
   *
   * @param int $entity_id
   *   The Drupal entity id.
   * @param string $bundle
   *   The Drupal entity bundle.
   * @param int $ext_id
   *   The external Bitrix 24 entity id.
   * @param string $ext_type
   *   The external Bitrix 24 entity type.
   * @param string $hash
   *   The hash string.
   */
  public function addReference(int $entity_id, string $bundle, int $ext_id, string $ext_type, string $hash = '') {
    $this->database->insert('b24_reference')
      ->fields([
        'entity_id' => $entity_id,
        'bundle' => $bundle,
        'ext_id' => $ext_id,
        'ext_type' => $ext_type,
        'hash' => $hash,
      ])->execute();
  }

  /**
   * Deletes  reference between Drupal and Bitrix24 entity.
   *
   * @param string $ext_type
   *   The Bitrix24 entity type id.
   * @param int $id
   *   The Bitrix24 entity id.
   * @param ?string $type
   *   Drupal referenced entity type.
   *
   * @return int
   *   The number of rows affected by the delete query.
   *
   * @throws \Exception
   */
  public function deleteReference(string $ext_type, int $id, ?string $type = NULL): int {
    $query = $this->database->delete('b24_reference')
      ->condition('ext_type', $ext_type)
      ->condition('ext_id', $id);
    if ($type) {
      $query->condition('bundle', $type);
    }
    return $query->execute();
  }

  /**
   * Updates a hash string.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   A Drupal entity earlier exported to Bitrix24.
   * @param string $ext_type
   *   Bitrix24 entity machine_name.
   * @param string $hash
   *   Base64 hash of fields in current state.
   *
   * @return int|null
   *   The number of rows matched by the update query.
   */
  public function updateHash(EntityInterface $entity, string $ext_type, string $hash): ?int {
    return $this->database->update('b24_reference')
      ->fields([
        'hash' => $hash,
      ])
      ->condition('bundle', $entity->getEntityTypeId())
      ->condition('ext_type', $ext_type)
      ->condition('entity_id', $entity->id())
      ->execute();
  }

  /**
   * Returns a hash string from a given data.
   *
   * @param array $fields
   *   Data to hash.
   *
   * @return string
   *   The hashed data.
   */
  public function getHash(array $fields): string {
    $string = serialize($fields);
    return Crypt::hashBase64($string);
  }

  /**
   * Returns list of already imported entities.
   *
   * @return array
   *   An array of found entities.
   */
  public function getExistingEntities(string $ext_type, $bundle): array {
    $existing_records = $this->database->select('b24_reference', 'r')
      ->fields('r', ['entity_id', 'ext_id'])
      ->condition('ext_type', $ext_type)
      ->condition('bundle', $bundle)
      ->execute()
      ->fetchAll();

    return array_column($existing_records, 'ext_id', 'entity_id');
  }

}
