<?php

namespace Drupal\b24_contact\Form;

use Drupal\contact\Entity\ContactForm;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\b24\Service\RestManager;

/**
 * Configure b24_contact settings for this site.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Drupal\b24\Service\RestManager definition.
   *
   * @var \Drupal\b24\Service\RestManager
   */
  protected $b24RestManager;

  /**
   * Constructs a new SettingsForm object.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    RestManager $b24_rest_manager
  ) {
    parent::__construct($config_factory);
    $this->b24RestManager = $b24_rest_manager;
    $this->configFactory = $config_factory;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('b24.rest_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'b24_contact_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['b24_contact.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('b24_contact.settings');

    $form['forms'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Contact form submissions exporting rules'),
    ];

    /** @var \Drupal\contact\Entity\ContactForm[] $contact_forms */
    $contact_forms = ContactForm::loadMultiple();

    foreach ($contact_forms as $contact_form_id => $contact_form) {

      $contact_form_settings = $config->get($contact_form_id);

      $form[$contact_form_id] = [
        '#type' => 'details',
        '#group' => 'forms',
        '#tree' => TRUE,
        '#title' => $contact_form->label(),
      ];

      $form[$contact_form_id]['status'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable export'),
        '#description' => $this->t('Export submissions of this form to Bitrix24 leads'),
        '#default_value' => $contact_form_settings['status'] ?? 0,
      ];

      $form[$contact_form_id]['mapping'] = [
        '#type' => 'details',
        '#title' => $this->t('Fields mapping'),
        '#states' => [
          'visible' => [
            '[name="' . $contact_form_id . '[status]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $this->getFields($form[$contact_form_id]['mapping'], $contact_form);
    }

    $form['#attached']['library'] = [
      'b24/vertical-tabs'
    ];

    return parent::buildForm($form, $form_state);
  }

  private function getFields(&$form, $contact_form) {
    $config = $this->config("b24_contact.settings")->get($contact_form->id());
    $b24_fields = $this->b24RestManager->getFields('lead');
    $elements = \Drupal::service('entity_field.manager')->getFieldDefinitions('contact_message', $contact_form->id());

    $restricted_types = [
      'uuid',
      'language',
      'entity_reference',
    ];
    $options = [];
    foreach ($elements as $element_id => $element) {
      if (in_array($element->getType(), $restricted_types)) {
        continue;
      }
      $options[$element_id] = $element->getLabel();
    }
    $options['custom'] = $this->t('Custom...');

    // todo: create tokens for submission values.
    $token_link = [
      '#theme' => 'token_tree_link',
      '#token_types' => ['site'],
      '#show_restricted' => TRUE,
      '#global_types' => TRUE,
    ];
    $token_link = \Drupal::service('renderer')->render($token_link);
    foreach ($b24_fields as $b24_field_name => $b24_field_properties) {
      if (in_array($b24_field_properties['type'], ['string', 'crm_multifield', 'double']) && !$b24_field_properties['isReadOnly']) {
        $form[$b24_field_name] = [
          '#type' => 'select',
          '#options' => $options,
          '#title' => $b24_field_properties['title'],
          '#default_value' => $config['mapping'][$b24_field_name] ?? '',
          '#empty_option' => $this->t('- None -'),
        ];
        if ($b24_field_properties['isRequired']) {
          $form[$b24_field_name]['#label_attributes']['class'][] = 'form-required';
        }
        $form["{$b24_field_name}_custom"] = [
          '#type' => 'textfield',
          '#title' => $this->t('@name custom value', ['@name' => $b24_field_properties['title']]),
          '#description' => $token_link,
          '#states' => [
            'visible' => [
              '[name="' . $contact_form->id() . '[mapping][' . $b24_field_name . ']"]' => [
                'value' => 'custom',
              ],
            ],
          ],
          '#default_value' => $config['mapping']["{$b24_field_name}_custom"] ?? '',
        ];
      }
    }

    return $b24_fields;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
    $form_state->cleanValues();
    $values = $form_state->getValues();
    foreach ($values as $form_id => $form_values) {
      if ($form_values['status']) {
        foreach ($form_values['mapping'] as $key => $value) {
          if (!$value && isset($form[$form_id]['mapping'][$key]['#label_attributes']['class']) && in_array('form-required', $form[$form_id]['mapping'][$key]['#label_attributes']['class'])) {
            $form_state->setErrorByName("$form_id][mapping][$key", t('@name field is required.', ['@name' => '«' . $form[$form_id]['mapping'][$key]['#title'] . '»']));
          }
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    $form_state->cleanValues();
    $config = $this->config('b24_contact.settings');
    foreach ($form_state->getValues() as $webform_id => $webform_settings) {
      $config->set($webform_id, $webform_settings);
    }
    $config->save();
  }

}
