<?php

namespace Drupal\b24_user\Service;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\b24\Service\RestManager;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\user\Entity\User;
use Drupal\Core\Utility\Token;
use Drupal\Core\Database\Connection;

/**
 * Class UserManager.
 */
class UserManager {

  /**
   * Drupal\Core\Config\ConfigFactoryInterface definition.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * An array of entities present in Bitrix24.
   * @var array
   */
  protected $existing;

  /**
   * An array of submitted values.
   *
   * @var array
   */
  protected $settings;

  /**
   * Drupal\b24\Service\RestManager definition.
   *
   * @var \Drupal\b24\Service\RestManager
   */
  protected $restManager;

  /**
   * Drupal\Core\Entity\EntityTypeManagerInterface definition.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a new UserManager object.
   */
  public function __construct(RestManager $b24_rest_manager,
                              EntityTypeManagerInterface $entity_type_manager,
                              ConfigFactoryInterface $config_factory,
                              Token $token,
                              Connection $connection) {
    $this->restManager = $b24_rest_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
    $this->token = $token;
    $this->database = $connection;
    $this->existing = $this->getExistingUsers();
  }

  public function getExistingUsers() {
    //return $this->restManager->getList('contact', ['select' => ['EMAIL']]);

    $existing_records = $this->database->select('b24_reference', 'r')
      ->fields('r', ['entity_id', 'ext_id'])
      ->condition('ext_type', 'contact')
      ->condition('bundle', 'user')
      ->execute()->fetchAll();
    $users = array_column($existing_records, 'ext_id', 'entity_id');

    return $users;
  }

  public function processUser($id, $update = FALSE) {
    $storage = $this->entityTypeManager->getStorage('user');
    /** @var \Drupal\user\Entity\User $user */
    $user = $storage->load($id);
    if (array_key_exists($user->id(), $this->existing) && !$update) {
      return;
    }
    $fields = $this->getValues($user);

    $existing_ids = $this->existing ?? [];
    $id = $existing_ids[$user->id()] ?? FALSE;
    $hash = $this->restManager->getHash($fields);
    if ($update && $id) {
      $fields = $this->getValues($user, $id);
      $this->restManager->updateContact($id, $fields);
      $this->restManager->updateHash($user, 'contact', $hash);
    }
    else {
      $id = $this->restManager->addContact($fields);
      \Drupal::database()->insert('b24_reference')
        ->fields([
          'entity_id' => $user->id(),
          'bundle' => 'user',
          'ext_id' => $id,
          'ext_type' => 'contact',
          'hash' => $hash,
        ])->execute();
    }

  }

  // todo: the following function is already has been implemented in in a similar way b24_commerce module thus it worth to be moved to a separate function of a parent module.
  public function getValues(User $user, $id = NULL) {
    $fields = array_filter($this->configFactory->get("b24_user.mapping")->get());
    $field_definitions = $this->configFactory->get("b24_user.field_types")->get();

    if ($id) {
      $existing_entity = $this->restManager->getContact($id);
    }

    foreach ($fields as $field_name => &$value) {
      $bubbleable_metadata = new BubbleableMetadata();
      $value = nl2br($this->token->replace($value, ['user' => $user], ['clear' => TRUE], $bubbleable_metadata));
      if ($field_definitions[$field_name]['isRequired'] && !$value) {
        $value = t('- Not set -');
      }
      /**
       * In Bitrix24 each item of a multiple field is a separate entity.
       * Updating multiple field items without mentioning id causes adding
       * another item. So we need to get all the related ids previously.
       */
      if ($field_definitions[$field_name]['type'] == 'crm_multifield') {
        $value = [['VALUE' => $value, 'VALUE_TYPE' => 'HOME']];
        if (!empty($existing_entity[$field_name])) {
          // todo: support multiple drupal fields.
          $value[0]['ID'] = reset($existing_entity[$field_name])['ID'];
        }
      }
    }
    return $fields;
  }

  public function importUser($data) {
    // todo: used existing fields mapping if possible (converting tokens to fields needed).
    //$fields = array_filter($this->configFactory->get("b24_user.mapping")->get());

    $storage = $this->entityTypeManager->getStorage('user');
    $query = $storage->getQuery();
    $group = $query->orConditionGroup();
    $group->condition('name', $data['NAME']);
    $group->condition('mail', reset($data['EMAIL'])['VALUE']);
    $query->condition($group);
    $query->range(0, 1);
    $ids = $query->execute();
    if(empty($ids)){
      $user = User::create([
        'name' => $data['NAME'],
        'mail' => reset($data['EMAIL'])['VALUE'],
        'roles' => $data['roles'],
        'status' => 1,
      ]);
      $save = $user->save();
      $fields = $this->getValues($user, $data['ID']);
      $hash = $this->restManager->getHash($fields);
      \Drupal::database()->insert('b24_reference')
        ->fields([
          'entity_id' => $user->id(),
          'bundle' => 'user',
          'ext_id' => $data['ID'],
          'ext_type' => 'contact',
          'hash' => $hash,
        ])->execute();
      return $save;
    }
    return FALSE;
  }


}
