<?php

namespace Drupal\b24_webform\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\webform\Entity\Webform;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\b24\Service\RestManager;

/**
 * Class SettingsForm.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Drupal\b24\Service\RestManager definition.
   *
   * @var \Drupal\b24\Service\RestManager
   */
  protected $b24RestManager;

  /**
   * Constructs a new SettingsForm object.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    RestManager $b24_rest_manager
    ) {
    parent::__construct($config_factory);
    $this->b24RestManager = $b24_rest_manager;
    $this->configFactory = $config_factory;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('b24.rest_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'b24_webform.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('b24_webform.settings');

    $form['webforms'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Webform submissions exporting rules'),
      //'#default_tab' => 'edit-convert_contact',
    ];


    $webforms = Webform::loadMultiple();
    /** @var  \Drupal\webform\Entity\Webform $webform */
    foreach ($webforms as $webform_id => $webform) {

      $form[$webform_id] = [
        '#type' => 'details',
        '#group' => 'webforms',
        '#tree' => TRUE,
        '#title' => $webform->label(),
      ];

      $webform_settings = $config->get($webform_id);

      $form[$webform_id]['status'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable export'),
        '#description' => $this->t('Export submissions of this webform to Bitrix24 leads'),
        '#default_value' => $webform_settings['status'] ?? 0,
      ];

      $form[$webform_id]['mapping'] = [
        '#type' => 'details',
        '#title' => $this->t('Fields mapping'),
        '#states' => [
          'visible' => [
            '[name="' . $webform_id . '[status]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $this->getFields($form[$webform_id]['mapping'], $webform);
    }

    $form['#attached']['library'] = [
      'b24/vertical-tabs'
    ];

    return parent::buildForm($form, $form_state);
  }

  private function getFields(&$form, $webform) {
    $config = $this->config("b24_webform.settings")->get($webform->id());
    $b24_fields = $this->b24RestManager->getFields('lead');

    $elements = $webform->getElementsDecodedAndFlattened();
    $restricted_types = [
      'webform_actions',
    ];
    $options = [];
    foreach ($elements as $element_id => $element) {
      if (in_array($element['#type'], $restricted_types)) {
        continue;
      }
      $options[$element_id] = $element['#title'];
    }
    $options['custom'] = $this->t('Custom...');
    $token_link = [
      '#theme' => 'token_tree_link',
      '#token_types' => ['site', 'webform', 'webform_submission'],
      '#show_restricted' => TRUE,
      '#global_types' => TRUE,
    ];
    $token_link = \Drupal::service('renderer')->render($token_link);
    foreach ($b24_fields as $b24_field_name => $b24_field_properties) {
      if (in_array($b24_field_properties['type'], ['string', 'crm_multifield', 'double']) && !$b24_field_properties['isReadOnly']) {
        $form[$b24_field_name] = [
          '#type' => 'select',
          '#options' => $options,
          '#title' => $b24_field_properties['title'],
          '#default_value' => $config['mapping'][$b24_field_name] ?? '',
          '#empty_option' => $this->t('- None -'),
        ];
        if ($b24_field_properties['isRequired']) {
          $form[$b24_field_name]['#label_attributes']['class'][] = 'form-required';
        }

        $form["{$b24_field_name}_custom"] = [
          '#type' => 'textfield',
          '#title' => $this->t('@name custom value', ['@name' => $b24_field_properties['title']]),
          '#description' => $token_link,
          '#states' => [
            'visible' => [
              '[name="' . $webform->id() . '[mapping][' . $b24_field_name . ']"]' => [
                'value' => 'custom',
              ],
            ],
          ],
          '#default_value' => $config['mapping']["{$b24_field_name}_custom"] ?? '',
        ];
      }
    }

    return $b24_fields;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
    $form_state->cleanValues();
    $values = $form_state->getValues();
    foreach ($values as $webform_id => $webform_values) {
      if ($webform_values['status']) {
        foreach ($webform_values['mapping'] as $key => $value) {
          if (!$value && isset($form[$webform_id]['mapping'][$key]['#label_attributes']['class']) && in_array('form-required', $form[$webform_id]['mapping'][$key]['#label_attributes']['class'])) {
            $form_state->setErrorByName("$webform_id][mapping][$key", t('@name field is required.', ['@name' => '«' . $form[$webform_id]['mapping'][$key]['#title'] . '»']));
          }
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    $form_state->cleanValues();
    $config = $this->config('b24_webform.settings');
    foreach ($form_state->getValues() as $webform_id => $webform_settings) {
      $config->set($webform_id, $webform_settings);
    }
    $config->save();
  }
}
