<?php

namespace Drupal\b24\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Core\Link;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\b24\Service\RestManager;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Core\State\State;

/**
 * Configure b24 settings for this site.
 */
class CredentialsForm extends ConfigFormBase {

  /**
   * @var \Drupal\b24\Service\RestManager
   */
  protected $restManager;

  /**
   * @var \Drupal\Core\State\State
   */
  protected $state;

  public function __construct(RestManager $rest_manager, RequestStack $request_stack, State $state) {
    $this->restManager = $rest_manager;
    $this->requestStack = $request_stack;
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('b24.rest_manager'),
      $container->get('request_stack'),
      $container->get('state')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'b24_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['b24.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $form['authorisation'] = array(
      '#type' => 'fieldset',
      '#title' => $this->t('Authorization'),
      '#collapsible' => true,
      '#collapsed' => true,
    );
    $form['authorisation']['site'] = array(
      '#type' => 'textfield',
      '#title' => $this->t('Bitrix24 site'),
      '#required' => TRUE,
      '#description' => $this->t('Type in your Bitrix24 domain (do not include "bitrix24.ru")'),
      '#default_value' => $this->config('b24.settings')->get('site'),
    );
    /*$form['authorisation']['login'] = array(
      '#type' => 'textfield',
      '#title' => $this->t('Login'),
      '#required' => TRUE,
      '#description' => $this->t('Type your Bitrix24 user login'),
      '#default_value' => $this->config('b24.settings')->get('login'),
    );
    $form['authorisation']['password'] = array(
      '#type' => 'password',
      '#title' => $this->t('Password'),
      '#description' => $this->t('Type in your Bitrix24 user password'),
      // This is a wrong way as it displays the password in the html.
      '#attributes' => ['value' => $this->config('b24.settings')->get('password')]
    );*/

    if ($this->config('b24.settings')->get('site')) {
      $form['authorisation']['app'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Application data'),
      ];

      $site_name = $this->config('b24.settings')
        ->get('site') ? $this->config('b24.settings')
        ->get('site') : '[your_site_name]';

      $app_add_link = Link::fromTextAndUrl("https://{$site_name}.bitrix24.ru/marketplace/local/edit/0/", Url::fromUri("https://{$site_name}.bitrix24.ru/marketplace/local/edit/0/", ['attributes' => ['target' => '_blank']]))
        ->toString();
      $app_list_link = Link::fromTextAndUrl("https://{$site_name}.bitrix24.ru/marketplace/local/list/", Url::fromUri("https://{$site_name}.bitrix24.ru/marketplace/local/list/", ['attributes' => ['target' => '_blank']]))
        ->toString();

      $form['authorisation']['app']['info'] = [
        '#type' => 'html_tag',
        '#tag' => 'div',
        '#attributes' => [
          'class' => ['messages'],
        ],
        '#value' => $this->t('You should create your application on your Bitrix24 portal: @link. Note: redirect_uri value must be @domain/b24/oauth', [
          '@link' => $app_add_link,
          '@domain' => $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost()
        ]),
      ];

      $form['authorisation']['app']['client_id'] = array(
        '#type' => 'textfield',
        '#title' => $this->t('Client ID'),
        '#description' => $this->t('Type in your Bitrix24 application client_id. If you have already created it you can find it on @link', ['@link' => $app_list_link]),
        '#default_value' => $this->config('b24.settings')->get('client_id'),
      );

      $form['authorisation']['app']['client_secret'] = array(
        '#type' => 'textfield',
        '#title' => $this->t('Client secret code'),
        '#description' => $this->t('Type in your Bitrix24 application client_secret. If you have already created it you can find it on @link', ['@link' => $app_list_link]),
        '#default_value' => $this->config('b24.settings')->get('client_secret'),
      );
    }

    if ($this->config('b24.settings')->get('client_secret')) {
      $token_request_link = $this->restManager->getAuthorizeUri();
      $form['authorisation']['app']['access_token'] = array(
        '#type' => 'textfield',
        '#title' => $this->t('Access token'),
        '#description' => $this->t('Click «@link» to request for a new one.', ['@link' => Link::fromTextAndUrl($this->t('Get access token'), Url::fromUri($token_request_link, ['attributes' => ['target' => '_blank']]))->toString()]),
        '#default_value' => $this->state->get('b24_access_token'),
        '#attributes' => [
          'disabled' => 'disabled',
        ],
      );
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('b24.settings');
    $config->set('site', $form_state->getValue('site'));

    // The latter two values (login+password) seem to be unneeded anymore.
    if ($form_state->getValue('login')) {
      $config->set('login', $form_state->getValue('login'));
    }
    if ($form_state->getValue('password')) {
      $config->set('password', $form_state->getValue('password'));
    }

    if ($form_state->getValue('client_id')) {
      $config->set('client_id', $form_state->getValue('client_id'));
    }
    if ($form_state->getValue('client_secret')) {
      $config->set('client_secret', $form_state->getValue('client_secret'));
    }
    /*if ($form_state->getValue('access_token')) {
      $config->set('access_token', $form_state->getValue('access_token'));
    }*/

    $config->save();
    parent::submitForm($form, $form_state);
  }

}
