<?php

namespace Drupal\b24\Service;

use GuzzleHttp\Exception\ClientException;
use Drupal\Core\Url;
use Drupal\user\Entity\User;
use Drupal\b24\Event\B24AssignEvent;
use Drupal\b24\Event\B24Event;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\State\State;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\ClientInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Logger\LoggerChannelFactory;
use Drupal\Core\Extension\ModuleHandler;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Drupal\Core\Database\Connection;
use Drupal\Component\Utility\Crypt;

/*$statuses = [
  'NEW' => 'Unassigned',
  'ASSIGNED' => 'Responsible Assigned',
  'DETAILS' => 'Waiting for Details',
  'CANNOT_CONTACT' => 'Cannot Contact',
  'IN_PROCESS' => 'In Progress',
  'ON_HOLD' => 'On Hold',
  'CONVERTED' => 'Converted',
  'JUNK' => 'Junk Lead',
];*/

/**
 * RestManager service.
 */
class RestManager {

  const AUTH_URI = '.bitrix24.ru/oauth/authorize';

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * @var \Drupal\Core\State\State
   */
  protected $state;

  /**
   * @var \Drupal\Core\Logger\LoggerChannelFactory $logger
   */
  protected $logger;

  /**
   * An http client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * \Drupal\Core\Extension\ModuleHandler
   */
  protected $moduleHandler;

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected $eventDispatcher;

  /**
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a restmanager object.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   */
  public function __construct(
    ConfigFactoryInterface $config,
    RequestStack $request_stack,
    State $state,
    LoggerChannelFactory $logger_factory,
    ClientInterface $http_client,
    ModuleHandler $module_handler,
    EventDispatcherInterface $event_dispatcher,
    Connection $connection
  ) {
    $this->config = $config->get('b24.settings');
    $this->defaultConfig = $config->get('b24.default_settings');
    $this->config_editable = $config->getEditable('b24.settings');
    $this->requestStack = $request_stack;
    $this->state = $state;
    $this->logger= $logger_factory->get('b24');
    $this->httpClient = $http_client;
    $this->moduleHandler= $module_handler;
    $this->eventDispatcher = $event_dispatcher;
    $this->database = $connection;
  }

  public function getAuthorizeUri() {
    $uri = 'https://' . $this->config->get('site') . self::AUTH_URI;
    $params = [
      'client_id' => $this->config->get('client_id'),
      'response_type' => 'code',
      'redirect_uri' => $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost() . '/b24/oauth'
    ];
    $query = UrlHelper::buildQuery($params);
    $uri = implode('?', [$uri, $query]);

    return $uri;
  }

  public function refreshAccessToken() {
    $uri = 'https://' . $this->config->get('site') . '.bitrix24.ru/oauth/token/';
    $scopes = ['crm'];
    $params = [
      'grant_type' => 'refresh_token',
      'client_id' => $this->config->get('client_id'),
      'client_secret' => $this->config->get('client_secret'),
      'scope' => implode(',', $scopes),
      'refresh_token' => $this->state->get('b24_refresh_token'),
      'redirect_uri' => $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost() . '/b24/oauth'
    ];
    $query = UrlHelper::buildQuery($params);
    $uri = implode('?', [$uri, $query]);
    try {
      $response = $this->httpClient->get($uri, array('headers' => array('Accept' => 'text/plain')));
      $data = $response->getBody();
      if (empty($data)) {
        $this->logger->error('Bitrix24 token refresh error: response data is empty.');
      }
      else {
        $data = Json::decode($data->__toString());
        if (!empty($data['access_token'])) {
          $token = $data['access_token'];
          $this->state->set('b24_access_token', $token);
          $this->state->set('b24_refresh_token', $data['refresh_token']);
          $this->state->set('b24_token_expires', $data['expires']);
          return $data;
        }
      }
    }
    catch (RequestException $e) {
      $this->logger->error('Bitrix24 token refresh error: @error.', ['@error' => $e->getMessage()]);
    }
  }

  /**
   * Base function for querying Bitrix24 REST API.
   *
   * @param string $method
   * @param $params
   * @return bool|mixed
   */
  public function get($method, $params = []) {
    $uri = 'https://' . $this->config->get('site') . ".bitrix24.ru/rest/$method/";
    $params['auth'] = $this->state->get('b24_access_token');
    $params = http_build_query($params);

    $uri = implode('?', [$uri, $params]);

    try {
      $request = $this->httpClient->get($uri);
      $this->refreshAccessToken();
      return Json::decode($request->getBody());
    }
    catch (ClientException $e) {
      $message = Json::decode($e->getResponse()->getBody()->getContents());
      $this->logger->error('Bitrix24 method «@error_name» error: @error_message.',
        ['@error_name' => $method,'@error_message' => $message['error_description']]);
      return FALSE;
    }
  }

  /**
   * Base function to create a Bitrix24 entity.
   *
   * @param string $b24_entity_name
   * @param array $fields
   * @param array $params
   * @return integer|bool
   */
  public function addEntity(string $b24_entity_name, $fields = [], $params = []) {
    $context = [
      'entity_name' => $b24_entity_name,
      'op' => 'insert',
    ];
    $this->moduleHandler->alter('b24_push', $fields, $context);
    if ($assignee_id = $this->defaultConfig->get('assignee')) {
      $assignee_id = ($assignee_id !== 'custom') ? $assignee_id : $this->defaultConfig->get('user_id');
      $fields['ASSIGNED_BY_ID'] = $assignee_id;
    }
    $response = $this->get("crm.{$b24_entity_name}.add", ['fields' => $fields, 'params' => $params]);
    $ext_id = FALSE;
    if (!empty($response['result'])) {
      $ext_id = $response['result'];
      $link = 'https://' . $this->config->get('site') . ".bitrix24.ru/crm/{$b24_entity_name}/details/{$ext_id}/";
      $this->moduleHandler->invokeAll("b24_{$b24_entity_name}_insert", $response);
      $this->logger->info('A new @entity added to Bitrix24 CRM with id=@id',
        [
          '@entity' => $b24_entity_name,
          '@id' => $ext_id,
          'link' => \Drupal::service('link_generator')->generate('view', Url::fromUri($link, ['attributes' => ['target' => '_blank']])),
        ]
      );
      $event = new B24Event($b24_entity_name, 'insert', $response);
      $this->eventDispatcher->dispatch(B24Event::ENTITY_INSERT, $event);
    }
    return $ext_id;
  }

  public function getLead($id) {
    $response = $this->get('crm.lead.get', ['id' => $id]);
    return $response['result'] ?? [];
  }

  public function addLead($fields = array(), $params = ['REGISTER_SONET_EVENT' => 'Y']) {
    if (!array_key_exists('SOURCE_ID', $fields)) {
      $fields['SOURCE_ID'] = 'WEB';
    }

    $fields['OPENED'] = 'Y';
    if ($uid = \Drupal::currentUser()->id()) {
      $contact = $this->getReference(User::load($uid), 'contact');
      if ($contact) {
        $fields['CONTACT_ID'] = $contact['ext_id'];
      }
    }
    return $this->addEntity('lead', $fields, $params);
  }

  public function updateLead($id, $fields = array(), $params = ['REGISTER_SONET_EVENT' => 'Y']) {
    return $this->updateEntity('lead', $id, $fields, $params);
  }

  public function updateEntity($b24_entity_name, $id, $fields = array(), $params = ['REGISTER_SONET_EVENT' => 'Y']) {
    $context = [
      'entity_name' => $b24_entity_name,
      'op' => 'update',
    ];
    $this->moduleHandler->alter('b24_push', $fields, $context);
    $response = $this->get("crm.{$b24_entity_name}.update", ['id' => $id,'fields' => $fields, 'params' => $params]);
    $event = new B24Event($b24_entity_name, 'update', $response);
    $this->eventDispatcher->dispatch(B24Event::ENTITY_UPDATE, $event);
    return $response;
  }

  public function getDeal($id) {
    $response = $this->get('crm.deal.get', ['id' => $id]);
    return $response['result'] ?? [];
  }

  public function addDeal($fields = array(), $params = ['REGISTER_SONET_EVENT' => 'Y']) {
    return $this->addEntity('deal', $fields, $params);
  }

  public function updateDeal($id, $fields = array(), $params = ['REGISTER_SONET_EVENT' => 'Y']) {
    $response = $this->get('crm.deal.update', ['id' => $id,'fields' => $fields, 'params' => $params]);
    return $response['result'] ?? FALSE;
  }


  public function addContact($fields = array(), $params = ['REGISTER_SONET_EVENT' => 'Y']) {
    return $this->addEntity('contact', $fields, $params);
  }

  public function updateContact($id, $fields = array(), $params = ['REGISTER_SONET_EVENT' => 'Y']) {
    $response = $this->get('crm.contact.update', ['id' => $id, 'fields' => $fields, 'params' => $params]);
    return $response['result'] ?? FALSE;
  }

  public function deleteContact($id) {
    $result = $this->deleteEntity($id, 'contact');
    if ($result) {
      $this->deleteReference('contact', $id);
    }
    return $result;
  }

  public function getContact($id) {
    $response = $this->get('crm.contact.get', ['id' => $id]);
    return $response['result'] ?? [];
  }

  public function getFields($entity_name) {
    $response = $this->get("crm.$entity_name.fields", []);
    return $response['result'] ?? [];
  }

  public function getList($entity_name, $params = []) {
    $results = [];
    $response = $this->get("crm.$entity_name.list", $params);
    $results = array_merge($results, $response['result']);
    if (!empty($response['next'])) {
      $params['start'] = $response['next'];
      $results = array_merge($results, $this->getList($entity_name, $params));
    }
    return $results ?? [];
  }

  public function setLeadProducts($id, $items) {
    return $this->setProductRows($id, 'lead', $items);
  }

  public function setDealProducts($id, $items) {
    return $this->setProductRows($id, 'deal', $items);
  }



  public function setProductRows($id, $entity_name, $items) {
    $params = [
      'id' => $id,
      'rows' => $items,
    ];
    $existing = $this->get("crm.$entity_name.productrows.get", ['id' => $id]);
    if ($existing['result']) {
      $response = $this->get("crm.$entity_name.productrows.set", $params);
      return $response['result'] ?? [];
    }
    else {
      return [];
    }
  }

  /**
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *  A Drupal entity earlier exported to Bitrix24
   * @param string $ext_type
   *  Bitrix24 entity machine_name
   * @return mixed
   */
  public function getReference(EntityInterface $entity, string $ext_type) {
    $reference = $this->database->select('b24_reference', 'b')
      ->fields('b', ['ext_id', 'hash'])
      ->condition('bundle', $entity->bundle())
      ->condition('ext_type', $ext_type)
      ->condition('entity_id', $entity->id())
      ->execute()->fetchAssoc();

    return $reference;
  }

  public function deleteReference($ext_type, $id) {
    $this->database->delete('b24_reference')
      ->condition('ext_type', $ext_type)
      ->condition('ext_id', $id)
      ->execute();
  }

  /**
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *  A Drupal entity earlier exported to Bitrix24
   * @param string $ext_type
   *  Bitrix24 entity machine_name
   * @param string $hash
   *  Base64 hash of fields in current state
   */
  public function updateHash(EntityInterface $entity, string $ext_type, string $hash) {
    $this->database->update('b24_reference')
      ->fields([
        'hash' => $hash,
      ])
      ->condition('bundle', $entity->bundle())
      ->condition('ext_type', $ext_type)
      ->condition('entity_id', $entity->id())
      ->execute();
  }

  public function getHash(array $fields) {
    $string = serialize($fields);
    return Crypt::hashBase64($string);
  }

  public function getId($id, $entity_name) {
    $list = $this->getList($entity_name, ['filter' => ['XML_ID' => $id]]);
    if ($list) {
      return $list[0]['ID'] ?? FALSE;
    }
  }

  // todo: add setReference method.

  public function deleteEntity($id, $b24_entity_name, $external = FALSE) {
    if ($external) {
      $id = $this->getId($id, $b24_entity_name);
      if (!$id) {
        return FALSE;
      }
    }
    $response = $this->get("crm.{$b24_entity_name}.delete", ['id' => $id]);
    $event = new B24Event($b24_entity_name, 'delete', $response);
    $this->eventDispatcher->dispatch(B24Event::ENTITY_DELETE, $event);
    $result = $response['result'] ?? FALSE;
    return $result;
  }

  public function deleteProduct($id, $external = FALSE) {
    return $this->deleteEntity($id, 'product', $external);
  }



}
