<?php

namespace Drupal\babel_brush\Form;

use Drupal\babel_brush\Service\BabelBrushService;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for searching and deleting source strings in Babel Brush.
 *
 * This form allows users to enter a keyword to search for source strings
 * and then delete selected strings from the search results.
 *
 * @package Drupal\babel_brush\Form
 */
class BabelBrushSearchForm extends FormBase {

  /**
   * An array of lid values.
   *
   * @var array
   */
  protected array $lids = [];

  /**
   * The search keyword.
   *
   * @var string
   */
  protected string $keyword = '';

  /**
   * The form array.
   *
   * @var array
   */
  protected array $form = [];

  /**
   * The form state interface.
   *
   * @var \Drupal\Core\Form\FormStateInterface
   */
  protected FormStateInterface $formState;

  /**
   * Constructs a BabelBrushSearchForm object.
   *
   * @param \Drupal\babel_brush\Service\BabelBrushService $babelBrushService
   *   The BabelBrush service.
   */
  public function __construct(protected BabelBrushService $babelBrushService) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('babel_brush.service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'babel_brush_search_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $this->form = $form;
    $this->formState = $form_state;

    $this->form['keyword'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Enter source string'),
      '#required' => TRUE,
    ];

    $this->form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Search'),
    ];

    $this->form['search_results'] = [
      '#weight' => 100,
    ];

    if ($this->formState->getTriggeringElement()) {
      $this->keyword = $this->formState->getValue('keyword');
      $results = $this->babelBrushService->getAllSourcesStringsByKeyword($this->keyword);

      if (!empty($results)) {
        $this->form['search_results']['select_all'] = [
          '#type' => 'button',
          '#value' => $this->t('Select all'),
          '#attributes' => [
            'class' => ['select-all-button'],
          ],
          '#prefix' => '<div class="select-buttons">',
          '#suffix' => '</div>'
        ];

        $this->form['search_results']['deselect_all'] = [
          '#type' => 'button',
          '#value' => $this->t('Deselect all'),
          '#attributes' => [
            'class' => ['deselect-all-button'],
          ],
          '#suffix' => '<br><br>'
        ];

        foreach ($results as $result) {
          $this->form['search_results']['results']['lid_' . $result->lid] = [
            '#type' => 'checkbox',
            '#title' => $result->source,
            '#default_value' => FALSE,
            '#prefix' => '<div>',
            '#suffix' => '</div>'
          ];

          if (!empty($result->context)) {
            $this->form['search_results']['results']['context_' . $result->lid] = [
              '#markup' => "<small>Context: $result->context</small>",
              '#prefix' => '<div>',
              '#suffix' => '</div>',
            ];
          }
        }

        $this->form['search_results']['actions']['submit'] = [
          '#type' => 'submit',
          '#value' => $this->t('Delete'),
          '#submit' => ['::deleteSourcesStrings'],
          '#attributes' => [
            'id' => 'delete-button',
            'style' => 'display: none;'
          ]
        ];
      }

      if (!isset($this->form['search_results']['results'])) {
        $this->form['search_results']['no_result'] = [
          '#prefix' => '<p>',
          '#suffix' => '</p>',
          '#markup' => $this->t('Sorry, no results found for this search')
        ];
      }
    }

    $this->form['#attached']['library'][] = 'babel_brush/select_toggle';

    return $this->form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->formState = $form_state;
    $this->formState->setRebuild(TRUE);
  }

  /**
   * Deletes the selected source strings.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function deleteSourcesStrings(array &$form, FormStateInterface $form_state): void {
    $this->formState = $form_state;
    $values = $form_state->getValues();

    foreach ($values as $key => $value) {
      if (str_contains($key, 'lid') and $value) {
        $this->lids[] = str_replace('lid_', '', $key);
      }
    }

    $this->babelBrushService->deleteSourceStringByLids($this->lids);
  }

}
