<?php

namespace Drupal\babel_brush\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Psr\Log\LoggerInterface;

/**
 * Provides services for Babel Brush operations.
 *
 * This service includes methods for retrieving source strings by keyword and
 * deleting source strings by their lids.
 */
class BabelBrushService {

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a BabelBrushService object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logFactory
   *   The logger channel factory.
   */
  public function __construct(
    protected Connection $database,
    protected LoggerChannelFactoryInterface $logFactory,
  ) {
    $this->logger = $logFactory->get('babel_brush');
  }

  /**
   * Retrieves all source strings that match a given keyword.
   *
   * @param string $keyword
   *   The keyword to search for.
   *
   * @return array
   *   An array of matching source strings.
   */
  public function getAllSourcesStringsByKeyword(string $keyword): array {
    $query = $this->database->select('locales_source', 'ls');
    $query->leftJoin('locales_target', 'lt', 'ls.lid = lt.lid');
    $query->fields('ls', ['lid', 'source', 'context']);

    $query->condition('ls.source', '%' . $this->database->escapeLike($keyword) . '%', 'LIKE');

    return $query->execute()->fetchAll();
  }

  /**
   * Deletes source strings by their lids.
   *
   * @param array $lids
   *   An array of lids of the source strings to be deleted.
   *
   * @return bool
   *   TRUE if the deletion was successful, FALSE otherwise.
   */
  public function deleteSourceStringByLids(array $lids): bool {
    try {
      $this->deleteSourceStringInLocalesSourceTable($lids);
      $this->deleteSourceStringInLocalesTargetTable($lids);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error($e->getTraceAsString());
      return FALSE;
    }
  }

  /**
   * Deletes source strings in the locales_source table.
   *
   * @param array $lids
   *   An array of lids of the source strings to be deleted.
   */
  protected function deleteSourceStringInLocalesSourceTable(array $lids): void {
    $this->database->delete('locales_source')
      ->condition('lid', $lids, 'IN')
      ->execute();
  }

  /**
   * Deletes source strings in the locales_target table.
   *
   * @param array $lids
   *   An array of lids of the source strings to be deleted.
   */
  protected function deleteSourceStringInLocalesTargetTable(array $lids): void {
    $this->database->delete('locales_target')
      ->condition('lid', $lids, 'IN')
      ->execute();
  }

}
