/**
 * @file
 * Back-2-Top functionality using vanilla JavaScript.
 */

(function (Drupal, drupalSettings) {
  'use strict';

  /**
   * Back-2-Top behavior.
   */
  Drupal.behaviors.back2Top = {
    attach: function (context, settings) {
      // Ensure we only initialize once
      if (context !== document) {
        return;
      }

      // Check if button already exists
      if (document.querySelector('.back-2-top')) {
        return;
      }

      var config = drupalSettings.back2Top || {};
      
      // Create the button
      var button = createButton(config);
      
      // Add button to the page
      document.body.appendChild(button);
      
      // Set up scroll event listener
      setupScrollListener(button);
      
      // Set up click event listener
      setupClickListener(button);
    }
  };

  /**
   * Create the back-2-top button element.
   */
  function createButton(config) {
    var button = document.createElement('button');
    button.className = 'back-2-top ' + (config.position || 'bottom-right');
    button.setAttribute('aria-label', 'Back to top');
    button.setAttribute('title', 'Back to top');
    
    // Set button styles
    button.style.width = (config.size || 50) + 'px';
    button.style.height = (config.size || 50) + 'px';
    button.style.backgroundColor = config.color || '#007cba';
    
    // Create button content based on image type
    var content = createButtonContent(config);
    button.appendChild(content);
    
    return button;
  }

  /**
   * Create button content based on configuration.
   */
  function createButtonContent(config) {
    var imageType = config.image_type || 'arrow';
    var content;
    
    switch (imageType) {
      case 'chevron':
        content = document.createElement('div');
        content.className = 'chevron';
        break;
        
      case 'triangle':
        content = document.createElement('div');
        content.className = 'triangle';
        break;
        
      case 'custom':
        if (config.custom_image) {
          content = document.createElement('img');
          content.className = 'custom-image';
          content.src = getCustomImageUrl(config.custom_image);
          content.alt = 'Back to top';
        } else {
          // Fallback to arrow if custom image not available
          content = document.createElement('div');
          content.className = 'arrow';
        }
        break;
        
      default: // arrow
        content = document.createElement('div');
        content.className = 'arrow';
        break;
    }
    
    return content;
  }

  /**
   * Get the URL for custom image.
   */
  function getCustomImageUrl(fileId) {
    // This would need to be populated by the PHP side
    // For now, return a placeholder that could be replaced by the module
    return '/sites/default/files/back_2_top/' + fileId;
  }

  /**
   * Set up scroll event listener to show/hide button.
   */
  function setupScrollListener(button) {
    var scrollThreshold = window.innerHeight;
    var ticking = false;
    
    function updateButton() {
      var scrollTop = window.pageYOffset || document.documentElement.scrollTop;
      var documentHeight = Math.max(
        document.body.scrollHeight,
        document.body.offsetHeight,
        document.documentElement.clientHeight,
        document.documentElement.scrollHeight,
        document.documentElement.offsetHeight
      );
      
      // Only show button if page is taller than viewport and user has scrolled
      var shouldShow = documentHeight > window.innerHeight && scrollTop > scrollThreshold;
      
      if (shouldShow) {
        button.classList.add('visible');
      } else {
        button.classList.remove('visible');
      }
      
      ticking = false;
    }
    
    function requestTick() {
      if (!ticking) {
        requestAnimationFrame(updateButton);
        ticking = true;
      }
    }
    
    // Initial check
    updateButton();
    
    // Listen for scroll events
    window.addEventListener('scroll', requestTick, { passive: true });
    
    // Listen for resize events (in case content height changes)
    window.addEventListener('resize', requestTick, { passive: true });
  }

  /**
   * Set up click event listener for smooth scrolling to top.
   */
  function setupClickListener(button) {
    button.addEventListener('click', function(e) {
      e.preventDefault();
      
      // Smooth scroll to top
      var startPosition = window.pageYOffset;
      var startTime = null;
      var duration = 500; // 500ms animation
      
      function animateScroll(currentTime) {
        if (startTime === null) {
          startTime = currentTime;
        }
        
        var timeElapsed = currentTime - startTime;
        var progress = Math.min(timeElapsed / duration, 1);
        
        // Easing function (ease-out)
        var easeOut = 1 - Math.pow(1 - progress, 3);
        
        window.scrollTo(0, startPosition * (1 - easeOut));
        
        if (timeElapsed < duration) {
          requestAnimationFrame(animateScroll);
        }
      }
      
      requestAnimationFrame(animateScroll);
    });
  }

})(Drupal, drupalSettings);